/*******************************************************************************
 * Copyright (c) 2022, 2024 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.diffusiondata.gateway.framework;

import java.util.NoSuchElementException;
import java.util.concurrent.CompletableFuture;

import com.diffusiondata.gateway.framework.GatewayApplication.ApplicationDetails;
import com.diffusiondata.gateway.framework.SinkHandler.SinkServiceProperties;
import com.diffusiondata.gateway.framework.SourceHandler.SourceServiceProperties;
import com.diffusiondata.gateway.framework.exceptions.ApplicationConfigurationException;
import com.diffusiondata.gateway.framework.exceptions.ApplicationInitializationException;
import com.diffusiondata.gateway.framework.exceptions.DiffusionClientException;
import com.diffusiondata.gateway.framework.exceptions.DiffusionSecurityException;

/**
 * Provides static methods for use with the Diffusion Gateway Framework.
 * <p>
 * For full details of the use of the framework see
 * {@link com.diffusiondata.gateway.framework Package Guide}.
 *
 * @author DiffusionData Limited
 */
public enum DiffusionGatewayFramework {

    /**
     * Internal type - do not use.
     */
    IMPLEMENTATION;

    @SuppressWarnings("PMD.SingularField") // False positive
    private Implementation delegate;

    DiffusionGatewayFramework() {

        if (!Boolean.getBoolean("diffusion.gateway.framework.testoverride")) {

            final String implementation =
                "com.diffusiondata.gateway.FrameworkImplementation";

            try {
                @SuppressWarnings("unchecked")
                final Class<? extends Implementation> clazz =
                    (Class<? extends Implementation>) this.getClass()
                        .getClassLoader().loadClass(implementation);

                delegate = clazz.newInstance();
            }
            catch (
                ClassNotFoundException | IllegalAccessException |
                InstantiationException e) {
                throw new IllegalArgumentException(e);
            }
        }
    }

    /**
     * Not for public use.
     */
    void setDelegate(Implementation delegate) {
        if (!Boolean.getBoolean("diffusion.gateway.framework.testoverride")) {
            throw new UnsupportedOperationException();
        }
        this.delegate = delegate;
    }

    /**
     * Called by an application to introduce a gateway framework application
     * implementation and start the framework.
     * <p>
     * This may only be called once per JVM.
     * <p>
     * When this is called, the framework will call back on the application to
     * determine its {@link GatewayApplication#getApplicationDetails application
     * details} and then {@link GatewayApplication#initialize initialize} the
     * application.
     * <p>
     * Once the application is initialized, the {@link GatewayFramework} and
     * the application will be started.
     *
     * During the start of the framework, a connection with the Diffusion server
     * will be established, and service types supported by the application
     * will be registered.
     * <p>
     * After the connection with the Diffusion server,
     * {@link GatewayApplication#start start} will be called on the application.
     * <p>
     * If the connection with the server fails to establish due to a terminal
     * error, such as an authentication error, the framework will throw the
     * corresponding exception and shut down.
     * <p>
     *
     * @param application the application
     * @return a reference to the framework
     * @throws ApplicationConfigurationException if the passed configuration or
     *         configuration obtained from the server has been found to be invalid
     *         or incompatible with the application;
     *
     * @throws DiffusionSecurityException if the application principal
     *         does not have sufficient permissions to perform the operation;
     *
     * @throws DiffusionClientException if some other exception has been returned from the
     *         Diffusion server via the Diffusion Client API.
     *         The cause will provide more detail.
     * @throws ApplicationInitializationException if the connection with
     *         the Diffusion server fails for any other reason or if initialization fails
     *         for any reason.
     */
    public static GatewayFramework start(
        GatewayApplication application)
        throws ApplicationInitializationException {

        return IMPLEMENTATION.delegate.start(application);
    }

    /**
     * Shuts down the framework. Closes the session with Diffusion and
     * terminates any running threads or processes.
     * <p>
     * However, the instance of {@link DiffusionGatewayFramework} created in the
     * JVM will not be removed and will be reused if the application is
     * restarted.
     * <p>
     * This is mostly useful in tests to explicitly clean up resources in the
     * framework.
     * <p>
     * This method does not need to be called within the Gateway application,
     * as shutting the application is managed by the framework.
     */
    public static void shutdown() {
        IMPLEMENTATION.delegate.shutdown();
    }

    /**
     * Factory for an {@link ApplicationDetails} builder.
     *
     * @return a new application details builder
     */
    public static ApplicationDetails.Builder newApplicationDetailsBuilder() {
        return IMPLEMENTATION.delegate.newApplicationDetailsBuilder();
    }

    /**
     * Factory for builders of {@link SourceServiceProperties}.
     *
     * @return a new source service properties builder
     */
    public static SourceServiceProperties.Builder
        newSourceServicePropertiesBuilder() {
        return IMPLEMENTATION.delegate.newSourceServicePropertiesBuilder();
    }

    /**
     * Factory for builders of {@link SinkServiceProperties}.
     *
     * @return a new sink service properties builder
     */
    public static SinkServiceProperties.Builder
        newSinkServicePropertiesBuilder() {
        return IMPLEMENTATION.delegate.newSinkServicePropertiesBuilder();
    }

    /**
     * Extracts parts of the Diffusion topic branch in the supplied source path
     * and constructs another topic path specified by the supplied mapping
     * function.
     * <p>
     * The supplied path will only be mapped if it is a simple Diffusion topic
     * path. If a Diffusion topic selector is supplied, an
     * <code>IllegalArgumentException</code> will be thrown.
     * <p>
     * The mapping function comprises <b>path fragments</b> and <b>path
     * directives</b>.
     * <p>
     * A <b>path fragment</b> is a section that is literally extracted as
     * it is.
     * It may contain the path separator character '/'.
     * <p>
     * A <b>path directive</b> is an expression of format {@literal <path
     * (start,length)>}. It is used to select parts of the source path to
     * be re-used in the mapped topic path. <code>start</code> is
     * mandatory and specifies the index of the source path to start from
     * and <code>length</code> specifies the number of parts to include
     * from <code>start</code> index. The <code>length</code> of the
     * expression is optional – if it is missing, or '0', or a number
     * that specifies a length longer than the end of the source path,
     * the selection extends to the end of the source path. The
     * <code>start</code> starts from 0, i.e., the index of the first
     * part of the source path is 0.
     * <p>
     * With the above definition, given a source path `a/b/c/d` and the mapping
     * function {@literal 'prefix/<path(1, 2)>'}, this method constructs the final topic
     * path `prefix/b/c`. Here, `prefix/` is a path fragment that is used as it
     * is. `1` is the index of the part in the source path to start from, which
     * is `b`, and `2` is the number of parts to include after the start index,
     * which includes `c`. Thus, the final mapped path is `prefix/b/c`.
     * <p>
     * Similarly, for the above source path, the mapping function {@literal '<path(1)>'}
     * constructs the final path `b/c/d`. Here, the <code>length</code> argument
     * is missing in the path directive. Hence, all following parts will be
     * included in the final mapped path.
     * <p>
     * Similarly, for the above source path, the mapping function {@literal '<path(1,10)>'}
     * constructs the final path `b/c/d`. Here, the <code>length</code> argument
     * in the path directive contains a larger value that covers all
     * parts after <code>start</code>. Hence, all following parts will be
     * included in the final mapped path.
     * <p>
     * The <code>start</code> and <code>length</code> in the path directive
     * should be positive numbers. If <code>start</code> fails to find
     * the corresponding part in the source path, a
     * <code>NoSuchElementException</code> will be thrown. For example, in the
     * given source path `a/b/c/d`, if the mapping function is {@literal '<path(5)>'},
     * there is no part at index 5 in the source path, which causes a
     * <code>NoSuchElementException</code>.
     *
     * @param path the Diffusion source topic path to map
     * @param mappingFunction the mapping function to apply
     * @return the final mapped topic path
     * @throws NoSuchElementException if either <code>start</code> or
     *                                <code>length</code> in the path directive
     *                                in the mapping function fails to find the
     *                                corresponding part in the source path
     *
     * @since 2.0
     */
    public static String mapDiffusionTopic(String path, String mappingFunction) {
        return IMPLEMENTATION.delegate.mapDiffusionTopic(path, mappingFunction);
    }

    /**
     * Do not use - Use static methods on {@link DiffusionGatewayFramework}.
     */
    public interface Implementation {

        /**
         * Do not use - Use {@link DiffusionGatewayFramework#start}.
         *
         * @param application the application
         * @return the framework
         */
        GatewayFramework start(
            GatewayApplication application)
            throws ApplicationInitializationException;

        /**
         * Do not use - Use
         * {@link DiffusionGatewayFramework#newApplicationDetailsBuilder}.
         *
         * @return new builder
         */
        ApplicationDetails.Builder newApplicationDetailsBuilder();

        /**
         * Do not use - Use
         * {@link DiffusionGatewayFramework#newSourceServicePropertiesBuilder}.
         *
         * @return new builder
         */
        SourceServiceProperties.Builder newSourceServicePropertiesBuilder();

        /**
         * Do not use - Use
         * {@link DiffusionGatewayFramework#newSinkServicePropertiesBuilder}.
         *
         * @return new builder
         */
        SinkServiceProperties.Builder newSinkServicePropertiesBuilder();

        /**
         * Do not use - Use
         * {@link DiffusionGatewayFramework#shutdown}.
         *
         * @return a CompletableFuture that completes when the application
         *     completes shutdown.
         */
        CompletableFuture<?> shutdown();

        /**
         * Do not use - Use
         * {@link DiffusionGatewayFramework#mapDiffusionTopic}.
         *
         * @param path            the topic path to map
         * @param mappingFunction the function to use to map the path to
         * @return the mapped topic path
         */
        String mapDiffusionTopic(String path, String mappingFunction);
    }
}
