/*******************************************************************************
 * Copyright (c) 2022,2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.diffusiondata.gateway.framework;

import java.util.concurrent.CompletableFuture;

/**
 * The interface that a polling source handler must implement.
 * <p>
 * The handler is {@link #poll polled} at the frequency defined by the
 * configured polling interval. The next poll is called after the interval time
 * has expired starting from when the previous poll completed.
 * <p>
 * A polling source handler will not be polled until the
 * {@link ServiceHandler#start start} method has been called. It will also not
 * be polled if the {@link ServiceHandler#pause pause} method has been called
 * and polling will not resume until the {@link ServiceHandler#resume resume}
 * method is called.
 * <p>
 * An object that implements this interface must be returned when the
 * {@link GatewayApplication#addPollingSource
 * GatewayApplication.addPollingSource} method is called. A different object
 * must be returned for each different service. The object must implement the
 * {@link #poll} method to publish any update(s) to Diffusion using the supplied
 * {@link Publisher}.
 *
 * @author DiffusionData Limited
 */
public interface PollingSourceHandler extends SourceHandler {

    /**
     * Polls the handler for a new value or values.
     * <p>
     * The handler should use its {@link Publisher} to publish all available
     * updates to Diffusion and then complete.
     * <p>
     * If when called, the handler is no longer able to operate then it should
     * use its {@link StateHandler} to report this to the framework and complete
     * successfully.
     * <p>
     * This method must return {@link CompletableFuture} that completes
     * successfully when the poll processing is complete. The method can choose
     * to initiate the processing in a separate thread and return immediately or
     * block until all processing is done.
     *
     * @return a completable future that completes successfully when all updates
     *         have been published to Diffusion or alternatively, a change of
     *         status has been reported to the framework.
     *         <p>
     *         If the task completes successfully, the CompletableFuture result
     *         should be null. The result type is any rather than Void to
     *         provide forward compatibility with future iterations of this API
     *         that may provide a non-null result with a more specific result
     *         type.
     *         <p>
     *         If the future completes exceptionally the framework will stop
     *         polling the handler and the service will go to a
     *         {@link ServiceState#PAUSED PAUSED} state.
     *         <p>
     *         If an unchecked exception is thrown, instead of returning a failed
     *         completable future, the exception will be caught and logged,
     *         but this will not have any effect on further poll calls.
     */
    CompletableFuture<?> poll();

}