/*******************************************************************************
 * Copyright (c) 2022, 2024 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.diffusiondata.gateway.framework;

import static java.util.concurrent.CompletableFuture.completedFuture;

import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import com.diffusiondata.gateway.framework.converters.PayloadConverter;
import com.diffusiondata.gateway.framework.exceptions.ApplicationInitializationException;
import com.diffusiondata.gateway.framework.exceptions.InvalidConfigurationException;

/**
 * The interface that a sink handler must implement.
 * <p>
 * A sink handler receives updates from the framework via its {@link #update
 * update} method. A sink handler will only receive updates when the service is
 * in an {@link ServiceState#ACTIVE ACTIVE} state.
 * <p>
 * A sink handler will not receive updates until the {@link ServiceHandler#start
 * start} method has been called. It will also not receive updates if the
 * {@link ServiceHandler#pause pause} method has been called and updates will
 * not resume until the {@link ServiceHandler#resume resume} method is called.
 * <p>
 * An object that implements this interface must be returned when the
 * {@link GatewayApplication#addSink GatewayApplication.addSink} method is
 * called. A different object must be returned for each different service. The
 * object must implement the {@link #update update} method to publish any
 * update(s) received from the framework to the back end sink.
 * <p>
 * The handler may also optionally return {@link SinkServiceProperties service
 * specific properties} via its {@link #getSinkServiceProperties} method. If it
 * does not then defaults will be assumed.
 *
 * @param <T> the type of object that is expected by the {@link #update} method.
 *        This should match the type that is output by the
 *        {@link PayloadConverter payload converter} in use for the
 *        service.
 *
 * @author DiffusionData Limited
 */
public interface SinkHandler<T> extends ServiceHandler {

    /**
     * This is used to set topic-specific properties for topics consumed by this
     * service.
     * <p>
     * Properties can be built using {@link SinkServiceProperties.Builder a
     * builder} which can be created using
     * {@link DiffusionGatewayFramework#newSinkServicePropertiesBuilder()}.
     * <p>
     * The default implementation returns null, which means that all defaults
     * are assumed for service properties. See the methods of
     * {@link SinkServiceProperties.Builder} for details about defaults.
     *
     * @return service properties or null to indicate defaults
     *
     * @throws InvalidConfigurationException possibly thrown by
     *         {@link SinkServiceProperties.Builder}
     */
    default SinkServiceProperties getSinkServiceProperties()
        throws InvalidConfigurationException {
        return null;
    }

    /**
     * Called by the framework to publish a new value received from Diffusion to
     * the sink.
     * <p>
     * One or more {@link PayloadConverter payload converters} defined for the service
     * will have been applied to the Diffusion value received from the topic to produce the sink
     * specific value.
     *
     * @param path the path of the Diffusion topic from where the value has been
     *        received
     *
     * @param value the final value to publish to the sink. This value could
     *              be of different type from the type of the Diffusion topic
     *              as it could have been converted using a converter
     *
     * @param topicProperties the properties of the Diffusion topic from which the
     *                  value has been received
     *
     * @return a completable future that completes successfully when the update
     *         has been performed.
     *         <p>
     *         If the update completes successfully, the CompletableFuture
     *         result should be null. The result type is any rather than Void to
     *         provide forward compatibility with future iterations of this API
     *         that may provide a non-null result with a more specific result
     *         type.
     *         <p>
     *         If the future completes exceptionally or the future fails to
     *         complete within the timeout specified in the configuration, the
     *         framework will {@link ServiceHandler#pause pause} the service. If
     *         the method throws an exception, it will be logged but the service
     *         will not be paused.
     */
    CompletableFuture<?> update(
        String path,
        T value,
        TopicProperties topicProperties);

    /**
     * Called when the sink service is subscribed to a Diffusion topic.
     * <p>
     * This can be used to perform any required operations when a subscription to
     * a Diffusion topic occurs.
     *
     * @param topicPath the Diffusion topic path
     *
     * @param topicProperties the Diffusion topic properties
     *
     * @return a completable future that completes successfully when any operations
     *         following topic subscription completes.
     *         <p>
     *         If it completes successfully, the CompletableFuture
     *         result should be null. The result type is any rather than Void to
     *         provide forward compatibility with future iterations of this API
     *         that may provide a non-null result with a more specific result
     *         type.
     *         <p>
     *         If the future completes exceptionally or throws an exception,
     *         it will be logged by the framework.
     */
    default CompletableFuture<?> onSubscription(
        String topicPath,
        TopicProperties topicProperties) {

        return completedFuture(null);
    }

    /**
     * Called when the sink service is unsubscribed from a Diffusion topic.
     * <p>
     * This can be used to perform any required operations when an unsubscription
     * from a Diffusion topic occurs.
     *
     * @param topicPath the Diffusion topic path
     *
     * @param topicProperties the Diffusion topic properties
     *
     * @param reason the reason for unsubscription
     *
     * @return a completable future that completes successfully when any operations
     *         following topic unsubscription completes.
     *         <p>
     *         If it completes successfully, the CompletableFuture
     *         result should be null. The result type is any rather than Void to
     *         provide forward compatibility with future iterations of this API
     *         that may provide a non-null result with a more specific result
     *         type.
     *         <p>
     *         If the future completes exceptionally or throws an exception,
     *         it will be logged by the framework.
     */
    default CompletableFuture<?> onUnsubscription(
        String topicPath,
        TopicProperties topicProperties,
        UnsubscriptionReason reason) {

        return completedFuture(null);
    }

    /**
     * Class of the value expected by this handler.
     * <p>
     * When adding a sink service, this is called by the framework to
     * validate that the configured payload
     * converter converts the value to the type expected by the handler.
     *
     * @return the class of the value expected by this handler.
     */
    Class<T> valueType();

    /**
     * The service properties that apply to the sink service.
     * <p>
     * Returned by the {@link SinkHandler#getSinkServiceProperties
     * getSinkServiceProperties} method.
     */
    interface SinkServiceProperties {

        /**
         * Returns the list of names of
         * {@link PayloadConverter payload
         * converters} that are assigned for this service, to be used for
         * converting the Diffusion topic values to application format.
         * <p>
         * If more than one payload converter is assigned to the service, the
         * list will contain the names of converters in the order they will
         * be executed.
         * <p>
         * @return list of payload converter names
         *
         * @see Builder#payloadConverter(String)
         * @see Builder#payloadConverter(String, Map)
         * @see Builder#payloadConverter(PayloadConverter)
         */
        List<String> getPayloadConverterNames();

        /**
         * Returns a boolean flag to specify whether the framework should
         * subscribe to the configured topic selector when a service is
         * started.
         * <p>
         *
         * @return a boolean flag to specify whether the framework should
         * subscribe to the configured topic selector as soon as a service is
         * started.
         * @see Builder#autoSubscribe(boolean)
         */
        boolean autoSubscribe();

        /**
         * A builder for sink service properties.
         * <p>
         * A new instance of such a builder can be created using
         * {@link DiffusionGatewayFramework#newSinkServicePropertiesBuilder}.
         */
        interface Builder {

            /**
             * Specifies a boolean flag to specify whether the framework
             * should subscribe to the configured topic selector when a
             * service is started.
             * <p>
             * Defaults to true, meaning topics are subscribed to as soon as
             * the service is started.
             * <p>
             * If this is set to false, the adapter writer is responsible for
             * subscribing to Diffusion topics using {@link
             * Subscriber#subscribe()} or {@link Subscriber#subscribe(String)}
             * methods as required.
             * <p>
             * The {@link Subscriber} instance is passed in the
             * {@link GatewayApplication#addSink(ServiceDefinition, Subscriber, StateHandler)}.
             * <p>
             *
             * @param autoSubscribe the boolean flag to specify whether the framework
             *                     should subscribe to the configured topic
             *                      selector when a service is started.
             *
             * @return this builder
             */
            Builder autoSubscribe(boolean autoSubscribe);

            /**
             * Specifies the name of a payload converter to use for this
             * service.
             * <p>
             * The framework will use a converter with the
             * supplied name and add to the list of converters for this service.
             * <p>
             * This method can be used instead of {@link #payloadConverter(String, Map)} or
             * {@link #payloadConverter(PayloadConverter)} to let the
             * framework instantiate the converter with the supplied name, if it
             * does not require any parameters.
             * <p>
             * This can be called multiple times to assign multiple
             * converters for the service.
             * <p>
             * One of the converters defined in {@link PayloadConverter}
             * can be explicitly specified if required.
             *
             * @see #build() for details on how supplied converters will be
             * validated and how converters will be executed
             *
             * @param name the converter name.
             *
             * @return this builder
             *
             * @throws InvalidConfigurationException if name is null or empty or a
             *         {@link PayloadConverter payload
             *         converter} with the given name could not be found.
             * @throws ApplicationInitializationException if the converter
             *         with the supplied name is not allowed to be used for the
             *         application
             */
            Builder payloadConverter(String name)
                throws InvalidConfigurationException;

            /**
             * Specifies the name of a payload converter and parameters the
             * converter requires to be initialized.
             * <p>
             * The framework will instantiate a converter with the supplied
             * details and add to the list of converters for this service.
             * <p>
             * This method can be used instead of {@link #payloadConverter(String)} or
             * {@link #payloadConverter(PayloadConverter)} to let the
             * framework use the converter with the supplied name and
             * parameters.
             * <p>
             * This can be called multiple times to assign multiple
             * converters for the service.
             * <p>
             * @see #build() for details on how supplied converters will be
             * validated and how converters will be executed
             *
             * @param name the converter name.
             * @param parameters the parameters required to instantiate the
             *                   converter.
             * @return this builder
             *
             * @throws InvalidConfigurationException if name is null or a
             *         {@link PayloadConverter payload
             *         converter} with the given name could not be found
             *
             * @throws ApplicationInitializationException if the converter
             *         with the supplied name is not allowed to be used for the
             *         application or the payload converter instantiation fails
             */
            Builder payloadConverter(String name, Map<String, Object> parameters)
                throws InvalidConfigurationException;

            /**
             * Specifies a payload converter to use for the service. This
             * converter instance will be added in the list of converters for
             * this service.
             * <p>
             * This method can be used instead of
             * {@link #payloadConverter(String)}
             * or {@link #payloadConverter(String, Map)} in cases where an
             * instantiated converter needs to be supplied.
             * <p>
             * This can be called multiple times to assign multiple
             * converters for the service.
             * <p>
             * @see #build() for details on how supplied converters will be
             * validated and how converters will be executed
             *
             * @param converter the converter.
             *
             * @return this builder
             *
             * @throws InvalidConfigurationException if converter is null
             *
             * @throws ApplicationInitializationException if the converter
             *         with the supplied name is not allowed to be used for the
             *         application
             */
            Builder payloadConverter(
                PayloadConverter<?, ?> converter)
                throws InvalidConfigurationException;

            /**
             * Creates a {@link SinkServiceProperties} instance with the current
             * settings of this builder.
             * <p>
             * If no payload converter has been specified, the framework uses
             * default payload converters according to the topic type for
             * received update.
             * <p>
             * If a single converter has been assigned for this service,
             * its input type should correspond to the Diffusion
             * topic type or the input type should be
             * `Object` which accepts any input data. This also applies to
             * the first converter in the collection, if multiple converters
             * have been assigned for the service.
             * <p>
             * If multiple converters are specified, they
             * will be called in order. This order will be determined by the order the
             * converters are added in the builder using
             * {@link #payloadConverter(String)} or
             * {@link #payloadConverter(String, Map)} or
             * {@link #payloadConverter(PayloadConverter)}. Thus,
             * the input and output types of the converters in the chain should
             * match their predecessor and follower. This means that the
             * input type of the second converter in the chain should be the
             * same as, or a superclass or superinterface of, the output type
             * of the previous converter, and so on.
             * <p>
             * For example, consider following to create a SinkServiceProperties:
             * <pre>
             *     newSinkServicePropertiesBuilder()
             *      .payloadConverter(converter1)
             *      .payloadConverter("converter2",parameters)
             *      .payloadConverter("converter3")
             *      .build();
             * </pre>
             * With the above code snippet, a list of converters will be created
             * containing 'converter1', "converter2", and "converter3" in order.
             * The input type of 'converter1' should be 'Object' or
             * compatible with any Diffusion
             * topic type, and its output type should match the input type of
             * 'converter2'. Similarly, the input type of 'converter2' should
             * match the output type of 'converter1', and the output type should
             * match the input type of 'converter3', and so on.
             *
             * @return service properties
             * @throws InvalidConfigurationException if the input type of the
             *         first payload converter is not specific to a Diffusion topic
             *         type or is 'Object' that accepts any input values.
             */
            SinkServiceProperties build() throws InvalidConfigurationException;
        }
    }

    /**
     * The reason for an unsubscription from a Diffusion topic.
     */
    enum UnsubscriptionReason {
        /**
         * The topic has been unsubscribed by the framework.
         */
        FRAMEWORK,

        /**
         * The topic has been removed from the Diffusion server.
         */
        REMOVAL,

        /**
         * The topic has been unsubscribed for some other reason -
         * see logs for more details.
         */
        OTHER
    }
}