/*******************************************************************************
 * Copyright (c) 2022,2025 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.diffusiondata.gateway.framework;

import static java.util.concurrent.CompletableFuture.completedFuture;

import java.util.concurrent.CompletableFuture;

/**
 * Base interface for all service handlers.
 * <p>
 * A service handler is a user implementation of the back end requirements of a
 * specific configured service (an instance of a {@link ServiceType}).
 * <p>
 * A service handler can be a {@link SourceHandler source handler} which
 * publishes updates to Diffusion, a {@link SinkHandler sink handler} which
 * receives updates from Diffusion for publishing to some back end sink, or a
 * {@link HybridHandler} which can both consume from Diffusion topics and
 * publish back to Diffusion topics.
 *
 * @author DiffusionData Limited
 */
public interface ServiceHandler {

    /**
     * Start the service.
     * <p>
     * This may be used for any initial connection requirements. It is called
     * once and once only by the framework to prepare the service.
     * <p>
     * The default implementation of this method simply returns a completed
     * future, although it should always be implemented for
     * {@link StreamingSourceHandler}s to enable updates to start being
     * streamed.
     * <p>
     * A {@link SourceHandler} or (@link HybridHandler} must not start
     * publishing updates until this has been called and completes successfully.
     * <p>
     * A source or hybrid handler publishes updates using the {@link Publisher}
     * provided to it when it was created.
     * <p>
     *
     * @return a CompletableFuture that completes when the service has
     *         successfully started.
     *         <p>
     *         If the task completes successfully, the CompletableFuture result
     *         should be null. The result type is any rather than Void to
     *         provide forward compatibility with future iterations of this API
     *         that may provide a non-null result with a more specific result
     *         type.
     *         <p>
     *         If the task fails, the CompletableFuture should complete
     *         exceptionally.
     *         <p>
     *         If the completable future completes exceptionally or any
     *         exception is thrown, the service will be started in a
     *         {@link ServiceState#PAUSED PAUSED} state
     */
    default CompletableFuture<?> start() {
        return completedFuture(null);
    }

    /**
     * Notification that the service has been paused.
     * <p>
     * This will be called if there has been a user request (i.e. from the
     * Diffusion Console) to pause the service, or if the framework has detected
     * that the service needs to be paused (e.g. because the framework has lost
     * its connection with the Diffusion server), or if an application error has
     * occurred. The {@link PauseReason reason} will indicate the precise
     * reason.
     * <p>
     * If this is called then a {@link StreamingSourceHandler} or
     * {@link HybridHandler} must stop publishing updates, a
     * {@link PollingSourceHandler} will no longer be polled for updates, and a
     * {@link SinkHandler} will no longer receive any updates until
     * {@link #resume} is called.
     * <p>
     * The default implementation of this method simply returns a completed
     * future.
     * <p>
     * @param reason the reason that the service is being paused
     *
     * @return a CompletableFuture that completes when service has successfully
     *         paused.
     *         <p>
     *         If the task completes successfully, the CompletableFuture result
     *         should be null. The result type is any rather than Void to
     *         provide forward compatibility with future iterations of this API
     *         that may provide a non-null result with a more specific result
     *         type.
     *         <p>
     *         If the task fails, the CompletableFuture should complete
     *         exceptionally.
     *         <p>
     *         If the completable future completes exceptionally or any
     *         exception is thrown, regardless of the failure, the service will
     *         be paused and the exception will be logged.
     */
    default CompletableFuture<?> pause(PauseReason reason) {
        return completedFuture(null);
    };

    /**
     * Notification that the service is being resumed.
     * <p>
     * This will be called if the service was paused for any reason and it may
     * now be resumed. It may have been paused by user request, because the
     * service reported an error or because the framework lost connection to the
     * Diffusion server.
     * <p>
     * The {@link ResumeReason reason} provided will be the last outstanding
     * reason to resume. For example, if there was a user request to pause a
     * service and that service later reported a {@link StateHandler.Status#RED
     * RED} status, followed by a {@link StateHandler.Status#GREEN GREEN} status
     * then unless the user had since requested resume then the service would
     * remain paused. Only when the user resume is requested would this method
     * be called with a reason of {@link ResumeReason#REQUESTED REQUESTED}.
     * <p>
     * The default implementation of this method simply returns a completed
     * future.
     * <p>
     * @param reason gives the reason that the service is being resumed
     *
     * @return a CompletableFuture that completes when service has successfully
     *         resumed.
     *         <p>
     *         If the task completes successfully, the CompletableFuture result
     *         should be null. The result type is any rather than Void to
     *         provide forward compatibility with future iterations of this API
     *         that may provide a non-null result with a more specific result
     *         type.
     *         <p>
     *         If the task fails, the CompletableFuture should complete
     *         exceptionally.
     *         <p>
     *         If the completable future completes exceptionally or any
     *         exception is thrown, the service will be paused with
     *         {@link PauseReason#APPLICATION_ERROR} reason and exception
     *         logged.
     */
    default CompletableFuture<?> resume(ResumeReason reason) {
        return completedFuture(null);
    };

    /**
     * Stop the service.
     * <P>
     * This will be called when a request has been received from the Diffusion
     * management console to remove the service, or when the whole application
     * is being stopped. It will only be called once.
     * <p>
     * This may be used to disconnect from any back end resources.
     * <p>
     * After this has been called no further interactions with the handler will
     * occur and no further interactions with the framework are permitted.
     * <p>
     * The default implementation of this method simply returns a completed
     * future, although it should always be implemented for
     * {@link StreamingSourceHandler}s to prevent further publishing to
     * Diffusion.
     *
     * @return a CompletableFuture that completes when service has successfully
     *         stopped.
     *         <p>
     *         If the task completes successfully, the CompletableFuture result
     *         will be null. The result type is any rather than Void to provide
     *         forward compatibility with future iterations of this API that may
     *         provide a non-null result with a more specific result type.
     *         <p>
     *         If the task fails, the CompletableFuture should complete
     *         exceptionally.
     *         <p>
     *         If the completable future completes exceptionally or any
     *         exception is thrown the service will still be stopped and the
     *         exception will be logged.
     */
    default CompletableFuture<?> stop() {
        return completedFuture(null);
    }

    /**
     * The reason that a service is being paused.
     */
    enum PauseReason {
        /**
         * The service is pausing as framework has become disconnected from
         * Diffusion.
         */
        DISCONNECTED,

        /**
         * The service is pausing on request from the Diffusion console or set
         * in config.
         */
        REQUESTED,

        /**
         * The service is pausing because an error was returned or notified by
         * the application, or a timeout has occurred waiting for a response
         * from the application via a service handler.
         */
        APPLICATION_ERROR
    }

    /**
     * The reason that a service is being resumed.
     */
    enum ResumeReason {

        /**
         * The service is resuming as an active connection with the Diffusion server
         * has been established.
         */
        CONNECTED,

        /**
         * The service is resuming as, connection with a server was lost, and
         * now it's recovered.
         */
        RECOVERED,

        /**
         * The service is resuming as resume has been requested from the
         * Diffusion Console.
         */
        REQUESTED,

        /**
         * The service is resuming because an application error has been
         * resolved (i.e. A {@link ServiceHandler} has reported a
         * {@link StateHandler.Status#GREEN GREEN} or
         * {@link StateHandler.Status#AMBER AMBER} status after having
         * previously reported a {@link StateHandler.Status#RED RED} status, or
         * having experienced a timeout).
         */
        APPLICATION_RECOVERED

    }
}