/*******************************************************************************
 * Copyright (c) 2024 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.diffusiondata.gateway.framework;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;

import com.diffusiondata.gateway.framework.SinkHandler.SinkServiceProperties;
import com.diffusiondata.gateway.framework.exceptions.DiffusionClientException;
import com.diffusiondata.gateway.framework.exceptions.DiffusionSecurityException;
import com.diffusiondata.gateway.framework.exceptions.ServiceStateException;

/**
 * A subscriber to be used to subscribe to or unsubscribe from Diffusion topics.
 *
 * This can be used to subscribe to Diffusion topics as required, if auto
 * subscription is disabled for the service by setting
 * {@link SinkServiceProperties#autoSubscribe()} to false.
 *
 * @author DiffusionData Limited
 * @since 2.0
 */
public interface Subscriber {
    /**
     * Request subscription to configured topics in the Diffusion server.
     *
     * @return a CompletableFuture that completes when a response is received
     *         from the Diffusion server after subscription.
     *         <p>
     *         If the task completes successfully, the CompletableFuture result
     *         will be null. The result type is any rather than Void to provide
     *         forward compatibility with future iterations of this API that may
     *         provide a non-null result with a more specific result type.
     *         <p>
     *         If the task fails, the CompletableFuture will complete
     *         exceptionally with a {@link CompletionException}. Common reasons
     *         for failure, listed by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *         <ul>
     *         <li>{@link DiffusionSecurityException} &ndash; if the application
     *         principal does not have sufficient permissions to perform the
     *         operation;
     *         <li>{@link DiffusionClientException} &ndash; if some other
     *         exception has been returned from the Diffusion server via the
     *         Diffusion Client API. The cause will provide more detail.
     *         <li>{@link ServiceStateException} &ndash; if the service state
     *         is incompatible with the operation.
     *         </ul>
     */
    CompletableFuture<?> subscribe();

    /**
     * Request subscription to passed topics in the Diffusion server.
     * <p>
     * @param topics the topics to subscribe to.
     *               This can be a single topic path or a Diffusion topic
     *               selector. A topic selector is an expression that
     *               identifies one or more topics.
     *               <p>
     *               See {@literal https://docs
     *               .diffusiondata.com/cloud/latest/manual/html/designguide/data
     *               /publication/topic_selector_unified.html} for more details.
     *
     * @return a CompletableFuture that completes when a response is received
     *         from the Diffusion server after subscription.
     *         <p>
     *         If the task completes successfully, the CompletableFuture result
     *         will be null. The result type is any rather than Void to provide
     *         forward compatibility with future iterations of this API that may
     *         provide a non-null result with a more specific result type.
     *         <p>
     *         If the task fails, the CompletableFuture will complete
     *         exceptionally with a {@link CompletionException}. Common reasons
     *         for failure, listed by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *         <ul>
     *         <li>{@link DiffusionSecurityException} &ndash; if the application
     *         principal does not have sufficient permissions to perform the
     *         operation;
     *         <li>{@link DiffusionClientException} &ndash; if some other
     *         exception has been returned from the Diffusion server via the
     *         Diffusion Client API. The cause will provide more detail.
     *         <li>{@link ServiceStateException} &ndash; if the service state
     *         is incompatible with the operation.
     *         </ul>
     * @throws IllegalArgumentException if topics is an invalid topic selector
     *         expression
     */
    CompletableFuture<?> subscribe(String topics);

    /**
     * Request unsubscription to topics from the Diffusion server.

     * @param topics the topics to unsubscribe from.
     *               This can be a single topic path or a Diffusion topic
     *               selector. A topic selector is an expression that
     *               identifies one or more topics.
     *               <p>
     *               See {@literal https://docs
     *               .diffusiondata.com/cloud/latest/manual/html/designguide/data
     *               /publication/topic_selector_unified.html} for more details.
     *
     * @return a CompletableFuture that completes when a response is received
     *         from the Diffusion server after unsubscription.
     *         <p>
     *         If the task completes successfully, the CompletableFuture result
     *         will be null. The result type is any rather than Void to provide
     *         forward compatibility with future iterations of this API that may
     *         provide a non-null result with a more specific result type.
     *         <p>
     *         If the task fails, the CompletableFuture will complete
     *         exceptionally with a {@link CompletionException}. Common reasons
     *         for failure, listed by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *         <ul>
     *         <li>{@link DiffusionClientException} &ndash; if some other
     *         exception has been returned from the Diffusion server via the
     *         Diffusion Client API. The cause will provide more detail.
     *         <li>{@link ServiceStateException} &ndash; if the service state
     *         is incompatible with the operation.
     *         </ul>
     * @throws IllegalArgumentException if topics is an invalid topic selector
     *         expression
     */
    CompletableFuture<?> unsubscribe(String topics);

    /**
     * Returns the configured {@code diffusionTopicSelector} in the framework
     * configuration of the service.
     * <p>
     * If not configured, a null value will be returned.
     * <p>
     * This method can be used to check if the {@code diffusionTopicSelector} is
     * configured by an application user.
     *
     * @return the configured {@code diffusionTopicSelector} in the framework
     *     configuration, or null if not configured.
     */
    String getConfiguredDiffusionTopicSelector();
}
