/*******************************************************************************
 * Copyright (c) 2022, 2024 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.diffusiondata.gateway.framework;

import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiFunction;

import com.diffusiondata.gateway.framework.SinkHandler.SinkServiceProperties;
import com.diffusiondata.gateway.framework.exceptions.ApplicationConfigurationException;
import com.diffusiondata.gateway.framework.exceptions.InvalidConfigurationException;

import io.micrometer.core.instrument.MeterRegistry;

/**
 * The interface that a Gateway Framework application must implement.
 * <p>
 * This is the interface that must be implemented by a Gateway Framework
 * application developer.
 * <p>
 * See {@link com.diffusiondata.gateway.framework Package Guide} for full
 * details of how to write a Gateway Framework application.
 *
 * @author DiffusionData Limited
 */
public interface GatewayApplication {

    /**
     * Returns the basic details about the application.
     * <P>
     * Such details may be built using a builder obtained from
     * {@link DiffusionGatewayFramework#newApplicationDetailsBuilder}.
     *
     * @return the application details
     *
     * @throws ApplicationConfigurationException possibly thrown by
     *         {@link ApplicationDetails.Builder}
     */
    ApplicationDetails getApplicationDetails()
        throws ApplicationConfigurationException;

    /**
     * Called to initialize the application.
     * <p>
     * The application context which provides configuration and other
     * resources for the application.
     * <p>
     * Details provided in {@link ApplicationContext} can be used to instantiate
     * necessary parameters and classes as required.
     * <p>
     * This method is called by the framework when
     * {@link DiffusionGatewayFramework#start(GatewayApplication)}
     * method is called.
     * <p>
     * The default implementation of this method does nothing.
     *
     * @param applicationContext the application context
     *
     * @throws ApplicationConfigurationException to indicate a configuration
     *         failure
     */
    default void initialize(ApplicationContext applicationContext)
        throws ApplicationConfigurationException {
    }

    /**
     * Starts the application.
     * <p>
     * This is called by the framework after it has successfully connected to
     * Diffusion, allowing the application to perform any initial processing.
     * <p>
     * The default implementation of this method simply returns a completed future.
     *
     * @return a CompletableFuture that completes when application has
     *         successfully started.
     *         <p>
     *         If the task completes successfully, the CompletableFuture result
     *         should be null. The result type is any rather than Void to
     *         provide forward compatibility with future iterations of this API
     *         that may provide a non-null result with a more specific result
     *         type.
     *         <p>
     *         If the task fails, the CompletableFuture should complete exceptionally.
     *         <p>
     *         If the completable future completes exceptionally or any
     *         exception is thrown, the application will fail to start.
     */
    default CompletableFuture<?> start() {
        return CompletableFuture.completedFuture(null);
    }

    /**
     * Adds a streaming source service to the application.
     * <p>
     * This will be called as a result of such a service being defined in the
     * configuration.
     * <p>
     * This must be implemented if the application supports any source streaming
     * services.
     * <p>
     * The application must return an instance of {@link StreamingSourceHandler}
     * that is for the exclusive use of the service. The handler should be
     * initialized with the provided {@link ServiceDefinition} (which includes
     * the service parameters) and use the provided {@link Publisher} to publish
     * streamed updates to Diffusion. It should also use the provided
     * {@link StateHandler} to report back to the gateway framework any change
     * in the status of the service.
     * <p>
     * The default implementation of this method throws an exception to the
     * framework indicating that it has not been implemented.
     *
     * @param serviceDefinition the definition of the service
     *
     * @param publisher a reference to a publisher that the service handler can
     *        use to publish updates to Diffusion
     *
     * @param stateHandler to be used to report the current status of the
     *        handler to the framework or enquire upon the current
     *        {@link ServiceState}
     *
     * @return a service handler for the service
     *
     * @throws InvalidConfigurationException if one or more configuration items
     *         passed in the definition is invalid
     */
    default StreamingSourceHandler addStreamingSource(
        ServiceDefinition serviceDefinition,
        Publisher publisher,
        StateHandler stateHandler)
        throws InvalidConfigurationException {
        throw new UnsupportedOperationException(
            "Streaming source not supported");
    }

    /**
     * Adds a polling source service to the application.
     * <p>
     * This will be called as a result of such a service being defined in the
     * configuration.
     * <p>
     * This must be implemented if the application supports any source polling
     * services.
     * <p>
     * The application must return an instance of {@link PollingSourceHandler}
     * that is for the exclusive use of the service. The handler should be
     * initialized with the provided {@link ServiceDefinition} (which includes
     * the service parameters) and use the provided {@link Publisher} to publish
     * updates to Diffusion when {@link PollingSourceHandler#poll polled} It
     * should also use the provided {@link StateHandler} to report back to the
     * gateway framework any change in the status of the service.
     * <p>
     * The default implementation of this method throws an exception to the
     * framework indicating that it has not been implemented.
     *
     * @param serviceDefinition the definition of the service
     *
     * @param publisher a publisher to be used to publish to Diffusion when
     *        polled
     *
     * @param stateHandler to be used to report the current status of the
     *        handler to the framework or enquire upon the current
     *        {@link ServiceState}
     *
     * @return a service handler for the service
     *
     * @throws InvalidConfigurationException if one or more configuration items
     *         passed in the definition is invalid
     */
    default PollingSourceHandler addPollingSource(
        ServiceDefinition serviceDefinition,
        Publisher publisher,
        StateHandler stateHandler)
        throws InvalidConfigurationException {
        throw new UnsupportedOperationException(
            "Polling source not supported");
    }

    /**
     * Adds a sink service to the application.
     * <p>
     * This will be called as a result of such a service being defined in the
     * configuration.
     * <p>
     * This must be implemented if the application supports any sink services.
     * <p>
     * The application must return an instance of {@link SinkHandler} that is
     * for the exclusive use of the service. The handler should be initialized
     * with the provided {@link ServiceDefinition} (which includes the service
     * parameters) and use the provided {@link StateHandler} to report back to
     * the gateway framework any change in the status of the service.
     * <p>
     * The default implementation of this method throws an exception to the
     * framework indicating that it has not been implemented.
     * <p>
     * @param serviceDefinition the definition of the service
     * <p>
     * @param subscriber a subscriber to be used to subscribe to
     *        Diffusion topics.
     *        <p>
     *        See {@link SinkServiceProperties.Builder#autoSubscribe(boolean)}
     * <p>
     * @param stateHandler to be used to report the current status of the
     *        handler to the framework or enquire upon the current
     *        {@link ServiceState}
     * <p>
     * @return a service handler for the service
     *
     * @throws InvalidConfigurationException if one or more configuration items
     *         passed in the definition is invalid
     */
    default SinkHandler<?> addSink(
        ServiceDefinition serviceDefinition,
        Subscriber subscriber,
        StateHandler stateHandler)
        throws InvalidConfigurationException {
        throw new UnsupportedOperationException("Sink not supported");
    }

    /**
     * Adds a hybrid service to the application.
     * <p>
     * This will be called as a result of such a service being defined in the
     * configuration.
     * <p>
     * This must be implemented if the application supports any hybrid
     * services.
     * <p>
     * The application must return an instance of {@link HybridHandler}
     * that is for the exclusive use of the service. The handler should be
     * initialized with the provided {@link ServiceDefinition} (which includes
     * the service parameters) and use the provided {@link Publisher} to publish
     * streamed updates to Diffusion. It should also use the provided
     * {@link StateHandler} to report back to the gateway framework any change
     * in the status of the service.
     * <p>
     * The default implementation of this method throws an exception to the
     * framework indicating that it has not been implemented.
     *
     * @param serviceDefinition the definition of the service
     * <p>
     * @param publisher a reference to a publisher that the hybrid handler can
     *        use to publish updates to Diffusion
     * <p>
     * @param subscriber a subscriber to be used to subscribe to
     *        Diffusion topics.
     *        <p>
     *        See {@link SinkServiceProperties.Builder#autoSubscribe(boolean)}
     * <p>
     * @param stateHandler to be used to report the current status of the
     *        handler to the framework or enquire upon the current
     *        {@link ServiceState}
     * <p>
     * @return a hybrid service handler for the service
     *
     * @throws InvalidConfigurationException if one or more configuration items
     *         passed in the definition is invalid
     */
    default HybridHandler<?> addHybrid(
        ServiceDefinition serviceDefinition,
        Publisher publisher,
        Subscriber subscriber,
        StateHandler stateHandler)
        throws InvalidConfigurationException {
        throw new UnsupportedOperationException(
            "Hybrid not supported");
    }

    /**
     * Stops the application.
     * <p>
     * This is called by the framework to indicate that it is to close down.
     * This allows the application to free any resources and tidy up.
     * <p>
     * The framework would have called the {@link ServiceHandler#stop} method on
     * each service handler before calling this.
     * <p>
     * The service will not be restarted once stopped.
     *
     * @return a CompletableFuture that completes when application has
     *         successfully stopped.
     *         <p>
     *         If the task completes successfully, the CompletableFuture result
     *         should be null. The result type is any rather than Void to provide
     *         forward compatibility with future iterations of this API that may
     *         provide a non-null result with a more specific result type.
     *         <p>
     *         If the task fails, the CompletableFuture should complete exceptionally.
     *         <p>
     *         If the completable future completes exceptionally or any
     *         exception is thrown, regardless of this exception, the
     *         exception will be logged and the application will terminate.
     */
    CompletableFuture<?> stop();

    /**
     * Initializes and returns an instance of {@link GatewayMeterRegistry} which should supply
     * {@link  MeterRegistry} for the application specific monitoring
     * tool/tools.
     * <p>
     * By default, null is returned, which means no metrics will be produced by the framework.
     * <p>
     * @param globalApplicationConfiguration the configured global parameters for
     *        the application. Set to null if metrics are not enabled in the
     *        global framework configuration.
     * <p>
     * @return {@link GatewayMeterRegistry} with application specific monitoring
     *     tool's {@link MeterRegistry}.
     * @see GatewayMeterRegistry#getMeterRegistry()
     */
    default GatewayMeterRegistry initializeGatewayMeterRegistry(
        Map<String, Object> globalApplicationConfiguration) {
        return null;
    }

    /**
     * Basic details of the application.
     * <P>
     * Such details must be returned from the {@link #getApplicationDetails}
     * method and may be built using a builder obtained from
     * {@link DiffusionGatewayFramework#newApplicationDetailsBuilder}.
     */
    interface ApplicationDetails {

        /**
         * A builder for application details.
         * <p>
         * A new instance of such a builder can be created using
         * {@link DiffusionGatewayFramework#newApplicationDetailsBuilder}.
         */
        interface Builder {

            /**
             * Sets the schema for the global application configuration.
             * <p>
             * The default if not specified is an empty schema.
             * <p>
             * If no schema is specified, no validation of the global
             * application configuration will be performed by the framework and
             * the application would not benefit from configuration formatting
             * in the Diffusion console. Not using a schema may be useful during
             * development but it is strongly advised to define a schema before
             * putting the application into production.
             *
             * @param schema a JSON schema (in draft 7 format) which describes
             *        the format of the global application configuration
             *
             * @return this builder
             *
             * @throws ApplicationConfigurationException if the schema is
             *         invalid
             */
            Builder globalSchema(String schema)
                throws ApplicationConfigurationException;

            /**
             * Add a service type.
             * <p>
             * This can be used to add a service type, if the
             * configuration schema for this service type is compatible
             * with any older configuration version and a configuration
             * upgrade function is not required.
             * <p>
             * If this builder method is used, by default, a bi-function
             * which returns the same configuration parameters is used as the
             * configuration upgrade function for this service type.
             * <p>
             * @see #addServiceType(String, ServiceMode, String, String, BiFunction) if such
             * function is required to be set.
             *
             * @param serviceTypeName the service type name
             *
             * @param mode the mode of operation of the service
             *
             * @param description the description of the service type.
             *        <p>
             *        This will be displayed in Diffusion console to provide
             *        details about the service type.
             *        <p>
             *        {@link ServiceMode#getDescription()} can be used as
             *        default or part of the description for the service type.

             * @param schema a JSON schema (in draft 7 format) which describes
             *        the format of the application specific configuration for
             *        the service type.
             *        <p>
             *        Null may be supplied to define an empty schema in which
             *        case no configuration validation will be performed by the
             *        framework and the application would not benefit from
             *        configuration formatting in the Diffusion console. Not
             *        using a schema may be useful during development but it is
             *        strongly advised to define a schema before putting the
             *        application into production.
             *
             * @return this builder
             *
             * @throws ApplicationConfigurationException if the supplied schema
             *         is invalid
             */
            Builder addServiceType(
                String serviceTypeName,
                ServiceMode mode,
                String description,
                String schema) throws ApplicationConfigurationException;

            /**
             * Add a service type.
             *
             * This can be used to add a service type, if the
             * configuration schema for this service type is incompatible
             * with any older configuration version and a configuration
             * upgrade function is required.
             *
             * @see #addServiceType(String, ServiceMode, String, String) if such
             * function is not required to be set.
             *
             * @param serviceTypeName the service type name
             *
             * @param mode the mode of operation of the service
             *
             * @param description the description of the service type.
             *        <p>
             *        This will be displayed in Diffusion console to provide
             *        details about the service type.
             *        <p>
             *        {@link ServiceMode#getDescription()} can be used as
             *        default or part of the description for the service type.
             *
             * @param schema a JSON schema (in draft 7 format) which describes
             *        the format of the application specific configuration for
             *        the service type.
             *        <p>
             *        Null may be supplied to define an empty schema in which
             *        case no configuration validation will be performed by the
             *        framework and the application would not benefit from
             *        configuration formatting in the Diffusion console. Not
             *        using a schema may be useful during development but it is
             *        strongly advised to define a schema before putting the
             *        application into production.
             *
             * @param configUpgradeFunction a {@link BiFunction} to upgrade
             *        configuration of the service type from older version to
             *        the latest supported version.
             *        <p>
             *        The bi-function would take the version of the configuration
             *        to be converted and the older configuration params as map,
             *        performs necessary transformation to upgrade the configuration
             *        and return upgraded configuration map.
             *
             * @return this builder
             *
             * @throws ApplicationConfigurationException if the supplied schema
             *         is invalid
             */
            Builder addServiceType(
                String serviceTypeName,
                ServiceMode mode,
                String description,
                String schema,
                BiFunction<
                    Integer,
                    Map<String, Object>,
                    Map<String, Object>> configUpgradeFunction)
                throws ApplicationConfigurationException;

            /**
             * Add a sharedConfig type.
             * <p>
             * This can be used to add a sharedConfig type, if the
             * configuration schema for this sharedConfig type is compatible
             * with any older configuration version and a configuration
             * upgrade function is not required.
             * <p>
             * @see #addSharedConfigType(String, String, String, BiFunction) if such
             * function is required to be set.
             * <p>
             * If this builder method is used, by default, a bi-function
             * which returns the same configuration parameters is used as the
             * configuration upgrade function for this sharedConfig type.
             * <p>
             * @param sharedConfigTypeName the sharedConfig type name
             *
             * @param description the description of the sharedConfig type
             *        <p>
             *        This will be displayed in Diffusion console to provide
             *        details about the sharedConfig type.
             *
             * @param schema a JSON schema (in draft 7 format) which describes
             *        the format of the application specific configuration for
             *        the sharedConfig type.
             *        <p>
             *        Null may be supplied to define an empty schema in which
             *        case no configuration validation will be performed by the
             *        framework and the application would not benefit from
             *        configuration formatting in the Diffusion console. Not
             *        using a schema may be useful during development, but it is
             *        strongly advised to define a schema before putting the
             *        application into production.
             *
             * @return this builder
             *
             * @throws ApplicationConfigurationException if the supplied schema
             *         is invalid
             */
            Builder addSharedConfigType(
                String sharedConfigTypeName,
                String description,
                String schema)
                throws ApplicationConfigurationException;

            /**
             * Add a sharedConfig type.
             *
             * This can be used to add a sharedConfig type, if the
             * configuration schema for this sharedConfig type is incompatible
             * with any older configuration version and a configuration
             * upgrade function is required.
             *
             * @see #addSharedConfigType(String, String, String) if such
             * function is not required to be set.
             *
             * @param sharedConfigTypeName the sharedConfig type name
             *
             * @param description the description of the sharedConfig type
             *        <p>
             *        This will be displayed in Diffusion console to provide
             *        details about the sharedConfig type.
             *
             * @param schema a JSON schema (in draft 7 format) which describes
             *        the format of the application specific configuration for
             *        the sharedConfig type.
             *        <p>
             *        Null may be supplied to define an empty schema in which
             *        case no configuration validation will be performed by the
             *        framework and the application would not benefit from
             *        configuration formatting in the Diffusion console. Not
             *        using a schema may be useful during development, but it is
             *        strongly advised to define a schema before putting the
             *        application into production.
             * @param configUpgradeFunction a {@link BiFunction} to upgrade
             *        configuration of the sharedConfig type from an older version to
             *        the latest supported version.
             *        <p>
             *        The bi-function would take the version of the configuration
             *        to be converted and the older configuration parameters as map,
             *        performs necessary transformation to upgrade the configuration
             *        and return upgraded configuration map.
             *
             * @return this builder
             *
             * @throws ApplicationConfigurationException if the supplied schema
             *         is invalid
             */
            Builder addSharedConfigType(
                String sharedConfigTypeName,
                String description,
                String schema,
                BiFunction<Integer, Map<String, Object>, Map<String, Object>> configUpgradeFunction)
                throws ApplicationConfigurationException;

            /**
             * Sets a {@link BiFunction} to upgrade global application
             * configuration from an older version to the latest version
             * supported by the application.
             * <p>
             * The bi-function would take the version of the configuration to be
             * converted and the older configuration parameters as a map,
             * perform the necessary transformation to the parameters and
             * return an upgraded configuration map.
             * <p>
             * The default if not specified is a bi-function which returns the
             * same configuration without any changes.
             * <p>
             * If the application supports usage of global configuration for
             * application and the schema of the configuration has changed from the
             * previous version with breaking changes, then this function should
             * be set, to upgrade passed global configuration of older version
             * to the latest expected version.
             * <p>
             * If the application does not support usage of global application
             * configuration, this bi-function does not need to be set.
             *
             * @param globalConfigUpgradeFunction A bi-function which will be
             *        used by framework to upgrade older global configuration to
             *        the latest version.
             *
             * @return this builder
             */
            Builder globalConfigUpgradeFunction(
                BiFunction<Integer, Map<String, Object>, Map<String, Object>> globalConfigUpgradeFunction);

            /**
             * Create an {@link ApplicationDetails} instance with the current
             * settings of this builder.
             *
             * @param applicationType specifies the application type. For
             *        example, an adapter that interfaces with Kafka might
             *        return type "Kafka".
             *
             * @param configurationVersion specifies the latest configuration
             *        version of the application.
             *        <p>
             *        Application configuration includes global configuration,
             *        configuration of sharedConfig types and service types defined by Gateway Application.
             *
             * @return application details
             */
            ApplicationDetails build(String applicationType, int configurationVersion);
        }

        /**
         * Returns the application type.
         *
         * @return the application type
         */
        String getApplicationType();

        /**
         * Returns the global configuration schema.
         *
         * @return the schema
         */
        String getGlobalSchema();

        /**
         * Returns the services types.
         * <p>
         * At least one will have been defined.
         *
         * @return list of supported service types
         */
        List<ServiceType> getServiceTypes();

        /**
         * Returns the sharedConfig types.
         * <p>
         * The use of sharedConfigs is optional. An empty list will be returned if
         * sharedConfigs are not in use.
         *
         * @return list of supported sharedConfig types
         */
        List<SharedConfigType> getSharedConfigTypes();

        /**
         * Returns the configuration version of application.
         *
         * @return configuration version of application.
         */
        int getConfigurationVersion();
    }
}