/*******************************************************************************
 * Copyright (c) 2022, 2024 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.diffusiondata.gateway.framework;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;

import com.diffusiondata.gateway.framework.SourceHandler.SourceServiceProperties;
import com.diffusiondata.gateway.framework.exceptions.DiffusionClientException;
import com.diffusiondata.gateway.framework.exceptions.DiffusionSecurityException;
import com.diffusiondata.gateway.framework.exceptions.IncompatibleConfigurationException;
import com.diffusiondata.gateway.framework.exceptions.JSONPatchException;
import com.diffusiondata.gateway.framework.exceptions.PayloadConversionException;
import com.diffusiondata.gateway.framework.exceptions.ServiceStateException;

/**
 * Allows updates to be published to Diffusion by {@link SourceHandler source
 * handlers} or {@link HybridHandler hybrid handlers}.
 * <p>
 * A instance of this type is provided by the framework when a
 * {@link SourceHandler} (either a {@link PollingSourceHandler} or
 * {@link StreamingSourceHandler}) or a {@link HybridHandler} is instantiated.
 * <p>
 * It provides methods for the handler to pass updates to the framework and on
 * to Diffusion.
 * <p>
 * The publishing methods on this interface should only be called after the
 * service has been successfully {@link ServiceHandler#start started} and must
 * not publish if the service is {@link ServiceHandler#pause paused}.
 * <p>
 * A {@link PollingSourceHandler} should only publish updates when it is
 * {@link PollingSourceHandler#poll polled}.
 * <p>
 * A {@link HybridHandler} should only publish updates when its
 * {@link HybridHandler#update update} method is called.
 * <p>
 * If any of the publishing methods complete exceptionally then the service
 * handler may choose to ignore (and possibly log) the error. However, it could
 * also pause the service using the {@link StateHandler}. Depending upon the
 * nature of the exception the handler may get paused anyway.
 * <p>
 * A {link ServiceStateException} will occur if the handler attempts to publish
 * when the service is not in an {@link ServiceState#ACTIVE ACTIVE} state.
 * <p>
 * A publisher instance is for the exclusive use of the service handler for
 * which it was created and may not be used across services or after the service
 * has been {@link ServiceHandler#stop stopped}.
 *
 * @author DiffusionData Limited
 */
public interface Publisher {

    /**
     * Publishes a new value to Diffusion.
     * <p>
     * The payload converter specified for the service will be applied to the
     * supplied value in order to convert it to a Diffusion topic value.
     *
     * The order of updates to this method will be preserved in the topic
     * updates. Therefore, this method should be called in the expected order
     * of updates for a topic path, if order should be maintained.
     *
     * @param path the topic path
     *
     * @param value the unconverted value
     *
     * @return a CompletableFuture that completes when a response is received
     *         from the Diffusion server.
     *         <p>
     *         If the task completes successfully, the CompletableFuture result
     *         will be null. The result type is any rather than Void to provide
     *         forward compatibility with future iterations of this API that may
     *         provide a non-null result with a more specific result type.
     *         <p>
     *         If the task fails, the CompletableFuture will complete
     *         exceptionally with a {@link CompletionException}. Common reasons
     *         for failure, listed by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *         <ul>
     *         <li>{@link DiffusionSecurityException} &ndash; if the application
     *         principal does not have sufficient permissions to perform the
     *         operation;
     *         <li>{@link DiffusionClientException} &ndash; if some other
     *         exception has been returned from the Diffusion server via the
     *         Diffusion Client API. The cause will provide more detail.
     *         <li>{@link ServiceStateException} &ndash; if the service state is
     *         incompatible with the operation.
     *         </ul>
     *
     * @throws PayloadConversionException if the supplied value could not be
     *         converted by the payload converter configured for the service, or
     *         the value type is incompatible with the payload converter
     *
     * @throws IllegalArgumentException if supplied path to publish update to is
     *         not valid
     */
    CompletableFuture<?> publish(String path, Object value)
        throws PayloadConversionException;

    /**
     * Publishes a new value to Diffusion.
     * <p>
     * This method operates in a similar manner to
     * {@link #publish(String, Object)}, but it also provides an option to
     * specify {@link TopicProperties} to be used for the supplied topic.
     * <p>
     * Use this method in preference over {@link #publish(String, Object)}
     * to create Diffusion topics of specific types.
     * <p>
     * An application user can set topic properties for topics to be
     * created by a source service in the service configuration or default
     * values are applied. A {@link TopicProperties} instance that contains
     * user-configured or default topic properties can be accessed by using the
     * {@link #getConfiguredTopicProperties()} method, which is available after
     * the service handler is started. Any topic properties in this
     * configured topic properties instance can be overridden using any of
     * the helper methods in {@link TopicProperties} and passed in this method.
     * <p>
     * The helper method in {@link TopicProperties}, such as
     * {@link TopicProperties#withTopicType(TopicType)}, returns an immutable
     * instance of `TopicProperties`. Hence, values should be overridden in
     * the last created `TopicProperties` instance to override multiple values.
     * <p>
     * An application user can also specify payload converters to be used for
     * the updates to be published. If payload converters are specified in
     * the service configuration, then any user defined configuration will
     * take precedence and the passed topicProperties will be ignored. The
     * configured payload converter will be used to convert the updates
     * passed with this method. In this case, the output type of the
     * converter used will define the type of Diffusion topic to be published.
     * <p>
     * If any payload converters are not specified by a user or in
     * {@link SourceServiceProperties.Builder#payloadConverter(String)}}, a
     * default converter to produce data of the supplied topic type will be
     * used and topic properties specified in this method will take precedence.
     * <p>
     * If the topic properties passed with a previously published topic path
     * changes at runtime, the previously created topic should be removed using
     * {@link #remove(String)} before publishing again.
     * <p>
     * @param path the topic path
     *
     * @param value the unconverted value
     *
     * @param topicProperties The topic properties to use to create the topic.
     *                        If many paths require the same
     *                        {@link TopicProperties} then the same instance
     *                        should be used for efficiency.
     *
     * @return a CompletableFuture that completes when a response is received
     *         from the Diffusion server.
     *         <p>
     *         If the task completes successfully, the CompletableFuture result
     *         will be null. The result type is any rather than Void to provide
     *         forward compatibility with future iterations of this API that may
     *         provide a non-null result with a more specific result type.
     *         <p>
     *         If the task fails, the CompletableFuture will complete
     *         exceptionally with a {@link CompletionException}. Common reasons
     *         for failure, listed by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *         <ul>
     *         <li>{@link DiffusionSecurityException} &ndash; if the application
     *         principal does not have sufficient permissions to perform the
     *         operation;
     *         <li>{@link DiffusionClientException} &ndash; if some other
     *         exception has been returned from the Diffusion server via the
     *         Diffusion Client API. The cause will provide more detail.
     *         <li>{@link ServiceStateException} &ndash; if the service state is
     *         incompatible with the operation.
     *         </ul>
     *
     * @throws PayloadConversionException if the supplied value could not be
     *         converted by the payload converter configured for the service, or
     *         the value type is incompatible with the payload converter, or
     *         if the supplied topic type does not match the value created by
     *         the payload converter in the service configuration.
     *
     * @throws IllegalArgumentException if the supplied path to publish the
     *         update to is not valid
     *
     * @since 2.0
     */
    CompletableFuture<?> publish(
        String path,
        Object value,
        TopicProperties topicProperties)
        throws PayloadConversionException;

    /**
     * Apply a JSON patch to a Diffusion topic value.
     * <p>
     * This method may be used only with {@link TopicType#JSON JSON} topics to
     * apply a patch to a Diffusion topic value.
     * <p>
     * Also this cannot be used if the service properties for the service
     * specify {@link UpdateMode#STREAMING}.
     *
     * @see #setInitialJSONValueForPatchUpdates(String, String) to set
     * initial value for a Diffusion JSON topic, to which subsequent patch updates are
     * to be applied
     *
     * @param path the topic path
     *
     * @param patch the JSON patch to apply
     *
     * @return a CompletableFuture that completes when a response is received
     *         from the Diffusion server.
     *         <p>
     *         If the task completes successfully, the CompletableFuture result
     *         will be null. The result type is any rather than Void to provide
     *         forward compatibility with future iterations of this API that may
     *         provide a non-null result with a more specific result type.
     *         <p>
     *         If the task fails, the CompletableFuture will complete
     *         exceptionally with a {@link CompletionException}. Common reasons
     *         for failure, listed by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *         <ul>
     *         <li>{@link DiffusionSecurityException} &ndash; if the application
     *         principal does not have sufficient permissions to perform the
     *         operation;
     *         <li>{@link JSONPatchException} &ndash; if the patch failed to
     *         apply;
     *         <li>{@link DiffusionClientException} &ndash; if some other
     *         exception has been returned from the Diffusion server via the
     *         Diffusion Client API. The cause will provide more detail.
     *         <li>{@link ServiceStateException} &ndash; if the service state is
     *         incompatible with the operation.
     *         </ul>
     *
     * @throws IncompatibleConfigurationException if the topic type in the
     *         service properties is not {@link TopicType#JSON JSON} or
     *         {@link UpdateMode#STREAMING} is defined
     *
     * @throws IllegalArgumentException if supplied path to publish update to is
     *         not valid
     */
    CompletableFuture<?> applyJSONPatch(String path, String patch)
        throws IncompatibleConfigurationException;

    /**
     * Remove a topic or topics.
     * <p>
     * This allows the {@link SourceHandler} or {@link HybridHandler} to remove
     * a Diffusion topic or topics that it may have previously created
     * regardless of any persistence policy in use. The topic could have been
     * created using a topic prefix configured by a user. Hence, this prefix
     * will be prepended to the passed topic before removing the topic.
     * <p>
     * Only topics that the application principal has sufficient permission to
     * remove will be removed.
     *
     * @param topics a single topic may be removed by simply specifying its
     *        path. By specifying a path followed by a single / all topics below
     *        the specified path will be removed. By specifying a path followed
     *        by // all topics below the path and the topic at the path will be
     *        removed.
     *
     * @return a CompletableFuture that completes when a response is received
     *         from the Diffusion server.
     *         <p>
     *         If the task completes successfully, the CompletableFuture result
     *         will be null. The result type is any rather than Void to provide
     *         forward compatibility with future iterations of this API that may
     *         provide a non-null result with a more specific result type.
     *         <p>
     *         If the task fails, the CompletableFuture will complete
     *         exceptionally with a {@link CompletionException}. Common reasons
     *         for failure, listed by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *         <ul>
     *         <li>{@link DiffusionSecurityException} &ndash; if the application
     *         principal does not have sufficient permissions to perform the
     *         operation;
     *         <li>{@link DiffusionClientException} &ndash; if some other
     *         exception has been returned from the Diffusion server via the
     *         Diffusion Client API. The cause will provide more detail.
     *         <li>{@link ServiceStateException} &ndash; if the service state is
     *         incompatible with the operation.
     *         </ul>
     * @throws IllegalArgumentException if {@code topics} is not a valid
     *         topic selector.
     */
    CompletableFuture<?> remove(String topics);

    /**
     * Sets the initial value for a Diffusion JSON topic, to which subsequent patch
     * updates will be applied.
     * <p>
     * This method will only register the initial value for path in memory,
     * which will be used when {@link #applyJSONPatch(String, String)} method
     * is called.
     * If {@link #applyJSONPatch(String, String)} fails because topic does
     * not exist, the topic will be created using the value set in this
     * method.
     *
     * <p>
     * This method should be used before {@link
     * #applyJSONPatch(String, String)} is called, so that if the topic to
     * send patch to, does not exist, framework will create a JSON topic with
     * specified JSON value.
     *
     * <p>
     * If this method is called multiple times, value set in last method call
     * will be applied.
     * <p>
     * If this method is not called before applying patch to a JSON topic,
     * and the topic does not exist, the topic will be created with '{}' as
     * initial value.
     * <p>
     * @param path Diffusion topic path to which initial value is to be set.
     *
     * @param jsonValue JSON string value to be set as initial value when
     *        creating the JSON topic.
     *
     * @throws IllegalArgumentException if supplied jsonValue is not valid
     *         JSON data
     */
    void setInitialJSONValueForPatchUpdates(String path, String jsonValue);

    /**
     * Register a {@link MissingTopicNotificationHandler} which will be notified if a
     * client subscription request matches no known topics and the selector
     * prefix of the subscription matches the specified branch of the topic
     * tree.
     * <p>
     * Ideally, this method in {@link Publisher} should be called when starting
     * the {@link SourceHandler} which contains the publisher.
     * <p>
     * The provided handler is called when another session subscribes a topic
     * selector which does not match any topics and the selector prefix of
     * the subscription matches the specified branch of the topic tree for
     * which the handler is registered.
     *
     * @param topicPath identifies a branch of the topic tree
     *
     * @param missingTopicNotificationHandler the handler to use for notifying
     *        topic subscription at or below the {@code topicPath} (unless there is
     *        another handler registered for a more specific topic path)
     *
     * @return a CompletableFuture that completes when the handler
     *         registration is complete.
     *
     *          <p>
     *         If the task completes successfully, the CompletableFuture result will
     *         be null. The result type is any rather than Void to provide forward
     *         compatibility with future iterations of this API that may provide a
     *         non-null result with a more specific result type.
     *
     *         <p>
     *         If the task fails, the CompletableFuture will complete exceptionally
     *         with a {@link CompletionException}. Common reasons for failure,
     *         listed by the exception reported as the {@link CompletionException#getCause()
     *         cause}, include:
     *         <ul>
     *         <li>{@link DiffusionSecurityException} &ndash; if the application
     *         principal does not have {@code REGISTER_HANDLER} permission;
     *
     *         <li>{@link DiffusionClientException} &ndash; if some other
     *         exception has been returned from the Diffusion server via the
     *         Diffusion Client API. The cause will provide more detail.
     *         </ul>
     *
     * @throws IllegalStateException if there is an attempt to add a handler for a
     *         topicPath, for which another handler is already registered,
     *         in same service.
     */
    CompletableFuture<?> addMissingTopicHandler(
        String topicPath,
        MissingTopicNotificationHandler missingTopicNotificationHandler);

    /**
     * Removes {@link MissingTopicNotificationHandler} registered for the
     * specified topicPath in the service.
     * <p>
     * This method can be used to remove handler registration and stop
     * getting missing topic notifications for the topic path.
     *
     * <p>
     * @param topicPath topic path for which
     *        {@link MissingTopicNotificationHandler} should be removed
     *
     * @return a CompletableFuture that completes when the registered handler is removed.
     *         <p>
     *         If the task completes successfully, the CompletableFuture result will
     *         be null. The result type is any rather than Void to provide forward
     *         compatibility with future iterations of this API that may provide a
     *         non-null result with a more specific result type.
     *         <p>
     *         If the task fails, the CompletableFuture will complete exceptionally
     *         with a {@link CompletionException}. Common reasons for failure,
     *         listed by the exception reported as the {@link CompletionException#getCause()
     *         cause}, include:
     *         <ul>
     *         <li>{@link DiffusionClientException} &ndash; if some other
     *         exception has been returned from the Diffusion server via the
     *         Diffusion Client API. The cause will provide more detail.
     *         </ul>
     */
    CompletableFuture<?> removeMissingTopicHandler(String topicPath);

    /**
     * Returns a {@link TopicProperties} with user-configured or default
     * topic properties for the service.
     * <p>
     * If this method is accessed before {@link SourceHandler#start()} is
     * called, it will return null.
     * <p>
     * @return A {@link TopicProperties} instance with user-configured or default values.
     */
    TopicProperties getConfiguredTopicProperties();
}