/*******************************************************************************
 * Copyright (c) 2022 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.diffusiondata.gateway.framework;

import java.util.Map;
import java.util.UUID;

/**
 * A service definition contains the details of a service required when
 * instantiating a {@link ServiceHandler} for the service.
 * <p>
 * It provides the application with the name of the service, the unique ID of
 * the service, the {@link ServiceType service type} and the
 * configured parameters for the service as well as a {@link SharedConfig}
 * for the service if one has been defined. This is passed to the
 * {@link GatewayApplication} factory methods when adding services.
 *
 * @author DiffusionData Limited
 */
public interface ServiceDefinition {

    /**
     * Returns the service name.
     * <p>
     * The service name uniquely identifies an instance of a service that is
     * configured according to the schema of the {@link ServiceType} named by
     * {@link #getServiceType}.
     *
     * @return the service name
     */
    String getServiceName();

    /**
     * Returns the service identifier.
     * <p>
     * The service id uniquely identifies an instance of a service that is
     * created by the framework. ID of a service instance is set when it is
     * created and changes when the service is updated. This can be used to
     * internally identify different service objects with same names.
     *
     * @return id of the service
     */
    UUID getId();

    /**
     * Returns the service type.
     * <p>
     * The service type will be one of those provided by the
     * {@link GatewayApplication} via its
     * {@link GatewayApplication#getApplicationDetails getApplicationDetails}
     * method.
     *
     * @return service type
     */
    ServiceType getServiceType();

    /**
     * Returns the {@link SharedConfig} for the service.
     * <p>
     * If the service configuration specified a sharedConfig name then this will
     * return the corresponding sharedConfig with its parameters.
     *
     * @return sharedConfig details or null if no sharedConfig has been configured for
     *         the service
     */
    SharedConfig getSharedConfig();

    /**
     * Returns the configured parameters for the service, defined in the
     * <b>application</b> section of the service configuration.
     * <p>
     * The parameters are passed as a map of String keys and Object values which
     * represent the application specific JSON configuration object.
     * <p>
     * The types of the map values will depend upon the configuration
     * structures. For a simple JSON object with all String values, the values
     * would be of type String. For other JSON primitive value types the value
     * types could be String, Integer, Double etc. For more complex JSON
     * structures the type could be List (for a JSON array) or Map (for a JSON
     * object} and these could have values of any of the previously mentioned
     * types.
     *
     * @return map of configured parameters
     */
    Map<String, Object> getParameters();
}