/*******************************************************************************
 * Copyright (c) 2022 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.diffusiondata.gateway.framework;

import java.util.Map;
import java.util.UUID;

/**
 * Defines a SharedConfig.
 * <p>
 * A SharedConfig defines a map of shared configuration information that can be
 * used by one or more services. Typically, an application writer will use
 * SharedConfig when more than one service can use the same sets of
 * configuration such as connection details for a back end data source or sink.
 * <p>
 * If a sharedConfig has been declared for a service then this will be passed as
 * part of the {@link ServiceDefinition} provided to the
 * {@link GatewayApplication} factory method used to instantiate a
 * {@link ServiceHandler}.
 *
 * @author DiffusionData Limited
 */
public interface SharedConfig {

    /**
     * Returns the sharedConfig name.
     * <p>
     * The sharedConfig name uniquely identifies an instance of a sharedConfig that is
     * configured according to the schema of the {@link SharedConfigType} named by
     * {@link #getType}.
     *
     * @return the sharedConfig name
     */
    String getName();

    /**
     * Returns the sharedConfig identifier.
     * <p>
     * The sharedConfig id uniquely identifies an instance of a sharedConfig that is
     * created by the framework. ID of a sharedConfig instance is set when it is
     * created and changes when the sharedConfig is updated. This can be used to
     * internally identify different sharedConfig instances with same names.
     *
     * @return id of the sharedConfig
     */
    UUID getId();

    /**
     * Returns the sharedConfig type.
     * <p>
     * The sharedConfig type defines the schema for the sharedConfig configuration
     * parameters. A sharedConfig type with the given name will have been declared
     * by the {@link GatewayApplication} via its
     * {@link GatewayApplication#getApplicationDetails getApplicationDetails}
     * method during startup.
     *
     * @return the sharedConfig type
     */
    String getType();

    /**
     * Returns the sharedConfig parameters as defined by the configuration.
     * <p>
     * The parameters are passed as a map of String keys and Object values which
     * represent the application specific JSON configuration object.
     * <p>
     * The types of the map values will depend upon the configuration
     * structures. For a simple JSON object with all String values, the values
     * would be of type String. For other JSON primitive value types the value
     * types could be String, Integer, Double etc. For more complex JSON
     * structures the type could be List (for a JSON array) or Map (for a JSON
     * object} and these could have values of any of the previously mentioned
     * types.
     *
     * @return map of sharedConfig parameters
     */
    Map<String, Object> getParameters();
}