/*******************************************************************************
 * Copyright (c) 2022, 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.diffusiondata.gateway.framework;

import java.util.List;
import java.util.Map;

import com.diffusiondata.gateway.framework.SinkHandler.SinkServiceProperties;
import com.diffusiondata.gateway.framework.converters.PayloadConverter;
import com.diffusiondata.gateway.framework.exceptions.ApplicationInitializationException;
import com.diffusiondata.gateway.framework.exceptions.InvalidConfigurationException;

/**
 * The base interface for source service handlers.
 * <p>
 * A source service handler may be a {@link StreamingSourceHandler} or a
 * {@link PollingSourceHandler}.
 * <p>
 * The handler publishes updates to Diffusion using a {@link Publisher} provided
 * to it when it is instantiated.
 * <p>
 * The handler may also optionally return {@link SourceServiceProperties service
 * specific properties} via its {@link #getSourceServiceProperties} method. If
 * it does not then defaults will be assumed. These properties allow the handler
 * to define details about the topics it wants to create and/or the data
 * conversions to use.
 *
 * @author DiffusionData Limited
 */
public interface SourceHandler extends ServiceHandler {

    /**
     * This is used to set topic-specific properties for topics created by this
     * service.
     * <p>
     * Properties can be built using a {@link SourceServiceProperties.Builder
     * builder} which can be created using
     * {@link DiffusionGatewayFramework#newSourceServicePropertiesBuilder()}.
     * <p>
     * The default implementation returns null, which means that all defaults
     * are assumed for service properties. See the methods of
     * {@link SourceServiceProperties.Builder} for details about defaults.
     *
     * @return service properties or null if default are to be assumed
     *
     * @throws InvalidConfigurationException possibly thrown by
     *         {@link SourceServiceProperties.Builder}
     */
    default SourceServiceProperties getSourceServiceProperties()
        throws InvalidConfigurationException {
        return null;
    }

    /**
     * The properties that apply to topics created by and published to by the
     * service.
     * <p>
     * Returned by the {@link SourceHandler#getSourceServiceProperties
     * getSourceServiceProperties} method.
     */
    interface SourceServiceProperties {

        /**
         * Returns the type of topic that the service will publish to.
         *
         * @return the topic type
         *
         * @see Builder#topicType
         */
        TopicType getTopicType();

        /**
         * Returns a list of names of {@link PayloadConverter payload
         * converters} that are to be used for converting the published
         * values to Diffusion format.
         * <p>
         * If more than one payload converter is assigned to the service, the
         * list will contain the names of converters in the order they will
         * be executed.
         *
         * @return list of payload converter names
         *
         * @see SinkServiceProperties.Builder#payloadConverter(String)
         * @see SinkServiceProperties.Builder#payloadConverter(String, Map)
         * @see SinkServiceProperties.Builder#payloadConverter(PayloadConverter)
         */
        List<String> getPayloadConverterNames();

        /**
         * Returns the update mode to be used for updating topics.
         *
         * @return the update mode
         *
         * @see Builder#updateMode
         */
        UpdateMode getUpdateMode();

        /**
         * A builder for source service properties.
         * <p>
         * A new instance of such a builder can be created using
         * {@link DiffusionGatewayFramework#newSourceServicePropertiesBuilder}.
         */
        interface Builder {

            /**
             * Sets the topic type.
             * <p>
             * If this is not set the value will be decided by the
             * {@link #build} method according to whether a payload converter
             * has been specified or not.
             * <p>
             * @see #build() for details on how supplied topic type and
             * converters will be validated
             *
             * @param type the topic type. Setting to null removes any
             *        previously specified topic type
             *
             * @return this builder
             */
            Builder topicType(TopicType type);

            /**
             * Specifies the name of a payload converter to use for this
             * service.
             * <p>
             * The framework will use a converter with
             * the supplied name and add to the list of converters for this service.
             * <p>
             * This method can be used instead of {@link #payloadConverter(String, Map)} or
             * {@link #payloadConverter(PayloadConverter)} to let the
             * framework use the converter with the supplied name, if it
             * does not require any instantiation parameters.
             * <p>
             * This can be called multiple times to assign multiple
             * converters for the service.
             * <p>
             * One of the converters defined in {@link PayloadConverter}
             * can be explicitly specified if required.
             *
             * @see #build() for details on how supplied topic type and
             * converters will be validated and how converters will be executed
             *
             * @param name the payload converter name.
             *
             * @return this builder
             *
             * @throws InvalidConfigurationException if name is null or empty or a
             *         {@link PayloadConverter payload
             *         converter} with the given name could not be found
             * @throws ApplicationInitializationException if the converter
             *         with the supplied name is not allowed to be used for the
             *         application
             */
            Builder payloadConverter(String name)
                throws InvalidConfigurationException;

            /**
             * Specifies the name of a payload converter and parameters the
             * converter requires to be initialized.
             * <p>
             * The framework will
             * instantiate a converter with  the supplied details and add to the
             * list of converters for this service.
             * <p>
             * This method can be used instead of {@link #payloadConverter(String)} or
             * {@link #payloadConverter(PayloadConverter)} to let the
             * framework use the converter with the supplied name and
             * parameters.
             * <p>
             * This can be called multiple times to assign multiple
             * converters for the service.
             * <p>
             * @see #build() for details on how supplied topic type and
             * converters will be validated and how converters will be executed
             *
             * @param name the converter name.
             * @param parameters the parameters required to instantiate the
             *                   converter.
             * @return this builder
             *
             * @throws InvalidConfigurationException if name is null or empty or a
             *         {@link PayloadConverter payload
             *         converter} with the given name could not be found
             *
             * @throws ApplicationInitializationException if the converter
             *         with the supplied name is not allowed to be used for the
             *         application or the payload converter instantiation fails
             */
            Builder payloadConverter(String name, Map<String, Object> parameters)
                throws InvalidConfigurationException;

            /**
             * Specifies a payload converter to use for this service. This
             * converter instance will be added in the list of converters for
             * this service.
             * <p>
             * This method can be used instead of
             * {@link #payloadConverter(String)}
             * or {@link #payloadConverter(String, Map)} in cases where an
             * instantiated converter needs to be supplied.
             * <p>
             * This can be called multiple times to assign multiple
             * converters for the service.
             * <p>
             * @see #build() for details on how supplied topic type and
             * converters will be validated and how converters will be executed
             *
             * @param converter the converter.
             * @return this builder
             *
             * @throws InvalidConfigurationException if converter is null
             *
             * @throws ApplicationInitializationException if the converter
             *         with the supplied name is not allowed to be used for the
             *         application
             */
            Builder payloadConverter(
                PayloadConverter<?, ?> converter)
                throws InvalidConfigurationException;

            /**
             * Sets the update mode to be used for topics published to by the
             * service.
             * <p>
             * The default, if not set is {@link UpdateMode#STREAMING STREAMING}.
             *
             * @param updateMode the update mode
             *
             * @return this builder
             */
            Builder updateMode(UpdateMode updateMode);

            /**
             * Create a {@link SourceServiceProperties} instance with the
             * current settings of this builder.
             * <p>
             * If no topic type has been specified, it will be derived from
             * the last payload converter in the list and if no
             * payload converter has been
             * specified it will be assumed to be {@link TopicType#JSON}.
             * <p>
             * If no payload converter is set standard conversions as described
             * in {@link com.diffusiondata.gateway.framework.converters
             * "Payload Converters"} will be used according to the chosen topic
             * type.
             * <p>
             *
             * If a single converter has been assigned for this service, its output
             * type should correspond to the Diffusion topic type, which must be
             * compatible with the topic type if it has already been
             * specified. This also applies to the last converter in the
             * collection, if multiple converters have been assigned for the service.
             * <p>
             * If multiple converters are specified, they will be called in
             * order. This order will be determined by the order the
             * converters are added in the builder using
             * {@link #payloadConverter(String)} or
             * {@link #payloadConverter(String, Map)} or
             * {@link #payloadConverter(PayloadConverter)}.  Thus,
             * the input and output types of the converters in the chain should
             * match their predecessor and follower. This means that the
             * input type of the second converter in the chain should be the
             * same as, or a superclass or superinterface of, the output type
             * of the previous converter, and so on.
             * <p>
             * For example, consider following to create a SourceServiceProperties:
             * <pre>
             *     newSourceServicePropertiesBuilder()
             *      .topicType(TopicType.JSON)
             *      .payloadConverter(converter1)
             *      .payloadConverter("converter2",parameters)
             *      .payloadConverter("converter3")
             *      .build();
             * </pre>
             * With the above code snippet, a list of converters will be created
             * containing 'converter1', "converter2", and "converter3" in order.
             * The output type of 'converter3' should match the 'TopicType.JSON'
             * topic type, and its input type should match the output type of
             * 'converter2'. Similarly, the input type of 'converter2' should
             * match the output type of 'converter1', and the output type should
             * match the input type of 'converter3', and so on.
             * <p>
             * @return service properties
             *
             * @throws InvalidConfigurationException if the topic type and
             *          output type of the last payload converter are
             *          incompatible or if the output type of the last
             *          payload converter is not specific to a Diffusion
             *          topic type.
             */
            SourceServiceProperties build() throws InvalidConfigurationException;
        }
    }
}