/*******************************************************************************
 * Copyright (c) 2022 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.diffusiondata.gateway.framework;

import java.util.concurrent.CompletableFuture;

/**
 * State Handler.
 * <p>
 * Each service is provided with a state handler which it can use to report its
 * current status at any time. It may also be used to enquire upon the current
 * {@link ServiceState state} of the service.
 * <p>
 * A state handler instance is for the exclusive use of the service handler for
 * which it was created and may not be used across services or after the service
 * has been {@link ServiceHandler#stop stopped}.
 *
 * @author DiffusionData Limited
 */
public interface StateHandler {

    /**
     * Indicates the type of status being reported when using the
     * {@link #reportStatus} method.
     */
    enum Status {
        /**
         * Green indicates that the service is running normally.
         */
        GREEN,

        /**
         * Amber indicates that the service is running but not fully.
         */
        AMBER,

        /**
         * Red indicates that a problem has occurred that means the service is
         * not functioning and is in need of recovery.
         */
        RED;
    }

    /**
     * Reports current status.
     * <p>
     * This should be used whenever the handler's status changes in some way.
     * For example, if some failure occurs within the handler or the
     * application, which renders the handler inoperative. It can also be used
     * to report that a service is now (at least partly) operational, from the
     * point of view of the application.
     * <p>
     * Reporting a {@link Status#RED RED} status for an
     * {@link ServiceState#ACTIVE ACTIVE} service will cause the framework to
     * assume that the handler is no longer operating normally and the
     * {@link ServiceHandler#pause} method will be called with a reason of
     * {@link ServiceHandler.PauseReason#APPLICATION_ERROR APPLICATION_ERROR}.
     * This changes the service state to {@link ServiceState#PAUSED PAUSED}
     * which is returned when pause completes.
     * <p>
     * Reporting a {@link Status#RED RED} status for a service that is not
     * {@link ServiceState#ACTIVE ACTIVE} will have no effect but the status
     * message provided will become the latest one exposed to the Diffusion
     * console.
     * <p>
     * Reporting a {@link Status#GREEN GREEN} or {@link Status#AMBER AMBER}
     * status will notify the framework that the handler is now (at least
     * partly) operational. Hence, if the {@link Status#RED RED} status was
     * previously reported, the {@link ServiceHandler#resume} method will be
     * called with a reason of {@link ServiceHandler.ResumeReason#APPLICATION_RECOVERED
     * APPLICATION_RECOVERED} and the state will become
     * {@link ServiceState#ACTIVE ACTIVE} when resume completes. However, the
     * service may still remain {@link ServiceState#PAUSED PAUSED} if it
     * had been previously paused by user request or if there is no connection
     * to the Diffusion server. The new service state is returned to indicate
     * whether this is the case. If there is no other reason for the service to
     * remain {@link ServiceState#PAUSED PAUSED} then the state will become
     * {@link ServiceState#ACTIVE ACTIVE}.
     * <p>
     * Reporting a {@link Status#GREEN GREEN} or {@link Status#AMBER AMBER}
     * status for a service that is already {@link ServiceState#ACTIVE ACTIVE}
     * will have no effect but the status message provided will become the
     * latest one exposed to the Diffusion console.
     * <p>
     * If the state returned is {@link ServiceState#ACTIVE ACTIVE} then
     * streaming services may start publishing updates again (if they were not
     * already) and polling services will start to be polled again.
     * <p>
     * If the state returned is not {@link ServiceState#ACTIVE ACTIVE} then
     * updates cannot be published until the service is
     * {@link ServiceHandler#resume resumed}.
     *
     * @param status the status
     *
     * @param title the status title
     *
     * @param description the status description
     *
     * @return a CompletableFuture that completes with the new
     *         {@link ServiceState ServiceState} once the service has paused or
     *         resumed based on the supplied {@link Status}
     */
    CompletableFuture<ServiceState> reportStatus(
        StateHandler.Status status,
        String title,
        String description);

    /**
     * Returns the current state of the service.
     *
     * @return the service state
     */
    ServiceState getState();
}