/*******************************************************************************
 * Copyright (c) 2022, 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.diffusiondata.gateway.framework;

import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;

/**
 * An application context for {@link GatewayApplication}.
 * <p>
 * It provides the application with the configured details of the {@link
 * GatewayApplication} and any other resources that the framework provides.
 * <p>
 * This is passed to the {@link GatewayApplication#initialize} method.
 *
 * @author DiffusionData Limited
 */
public interface ApplicationContext {

    /**
     * Returns the application identifier as supplied in the configuration.
     *
     * @return application id
     */
    String getId();

    /**
     * Returns a boolean value to represent whether metrics is enabled in the
     * configuration, by application user.
     * <p>
     * This can be used to check whether components required to expose metrics
     * should be instantiated.
     * <p>
     * By default, exposing metrics is disabled in gateway application.
     *
     * @return a boolean value to represent whether metrics is enabled in
     * configuration.
     *
     * @since 2.0
     */
    boolean isMetricsEnabled();

    /**
     * Returns the configured global parameters for the application.
     * <p>
     * The parameters are passed as a map of String keys and Object values which
     * represent the application specific JSON configuration object.
     * <p>
     * The types of the map values will depend upon the configuration
     * structures. For a simple JSON object with all String values, the values
     * would be of type String. For other JSON primitive value types the value
     * types could be String, Integer, Double etc. For more complex JSON
     * structures the type could be List (for a JSON array) or Map (for a JSON
     * object} and these could have values of any of the previously mentioned
     * types.
     *
     * @return map of configured parameters
     */
    Map<String, Object> getGlobalConfiguration();

    /**
     * Returns instance of {@link ScheduledExecutorService} initialized by the
     * framework using the thread pool configured in configuration file.
     * <p>
     * This can be used to schedule processing in separate threads if required
     * by service handlers.
     * <p>
     * When shutting down the application, this thread pool will be terminated
     * by the framework.
     *
     * @return {@link ScheduledExecutorService} managed by framework.
     */
    ScheduledExecutorService getExecutorService();
}