/*******************************************************************************
 * Copyright (c) 2024 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.diffusiondata.gateway.framework;

import static java.util.Objects.requireNonNull;

import java.util.Map;
import java.util.concurrent.CompletableFuture;

import com.diffusiondata.gateway.framework.converters.PayloadConverter;
import com.diffusiondata.gateway.framework.exceptions.IncompatibleConfigurationException;
import com.diffusiondata.gateway.framework.exceptions.PayloadConversionException;

/**
 * An extension of a {@link Publisher} that enables overriding the default
 * publication mechanism in a specific type of Gateway application.
 * <p>
 * Normally, within a source or hybrid {@link ServiceHandler} in a Gateway
 * application, the {@link Publisher#publish(String, Object)} method is called to
 * publish an update to a Diffusion topic. This default behaviour can be overridden by
 * configuring the service to use a {@link CustomPublisher}. An implementation of
 * {@link CustomPublisher} can be plugged into the application to process updates and
 * perform operations provided by the {@link Publisher}. For example, it can divide a single
 * update into multiple updates, enrich those updates, publish them to multiple Diffusion topics,
 * or even remove a Diffusion topic based on the update.
 * <p>
 * Using a {@link CustomPublisher} is an advanced feature and is useful when
 * the default publication mechanism in an existing application needs to be overridden.
 * All Gateway applications supplied by <b>DiffusionData Limited</b> support this feature, allowing
 * developers to override the default publication. Externally developed applications
 * that need to process an update can simply do so by using a {@link Publisher}.
 * <p>
 * The Gateway application that supports custom publication should implement the
 * {@link UpdateContext} and provide additional getters for any context information
 * related to the update that will be supplied to the {@link CustomPublisher}.
 * The context should be documented clearly so that a {@link CustomPublisher} developer
 * understands the context supplied with the update and can use it as needed.
 * A {@link CustomPublisher} developer should include the Gateway application JAR as a dependency
 * to access the application's implemented {@link UpdateContext}. In Maven, the JAR can be
 * downloaded and included in the POM file with the <b>system</b> scope.
 * <p>
 * When a custom publisher is configured for a service, any {@link PayloadConverter}s defined
 * for the service will be ignored, as a custom publisher may process updates into different
 * formats and publish them to different Diffusion topics. However, the custom publisher itself
 * can still use {@link PayloadConverter}s if necessary, to convert updates before publishing them.
 * To receive the payload converter instances from the framework, the constructor of the
 * custom publisher should accept a Map of payload converter names to payload converter instances
 * as a constructor argument. Refer to the constructor documentation for further details.
 * <p>
 * Similar to a {@link PayloadConverter}, the implementation of a {@link CustomPublisher} can be
 * provided as a standalone JAR. This JAR can then be used with a Gateway application by
 * including it in the classpath when starting the application. This setup works if both the
 * application and the {@link CustomPublisher} are compatible. Specifically, the application
 * must support using a {@link CustomPublisher}, and the {@link CustomPublisher} must be capable
 * of handling the updates provided by the application.
 * <p>
 * If a {@link CustomPublisher} implementation is available in the classpath and a user has
 * configured a service to use it, the framework will pass an instance of the custom publisher
 * to the `add` methods of the {@link GatewayApplication} for source and hybrid service types.
 * This instance should be cast to a {@link CustomPublisher} and used to process the update
 * received by the source or hybrid service.
 *
 * @param <U> the type of the update to process
 * @param <C> the {@link UpdateContext} type to process the update
 *
 * @author DiffusionData Limited
 * @see Publisher
 * @since 2.2
 */
public abstract class CustomPublisher<U, C extends UpdateContext> implements Publisher {

    private final Publisher publisher;

    /**
     * Constructor to create an instance of {@link CustomPublisher}.
     * <p>
     * The Framework uses reflection to instantiate the subclass by injecting a
     * {@link Publisher} instance into the constructor of the subclass. The
     * subclass should supply this instance of the publisher to its super
     * constructor. The instance of the subclass thus created will then be
     * supplied to the `add` methods of the {@link GatewayApplication} for
     * source and hybrid service types, which will then be used to process the
     * updates.
     * <p>
     * If the subclass requires a {@link Map} of context for its construction, the
     * signature of the constructor can also include a Map, after Publisher,
     * containing keys of {@link String} type and values of {@link Object} type.
     * The framework then extracts the configured context for the
     * CustomPublisher in the service configuration and supplies it to the
     * constructor.
     * <p>
     * If the subclass also requires {@link PayloadConverter}s to process
     * updates, it can define a constructor that accepts a
     * {@link Publisher}, a {@link Map} of context, and a {@link Map} of
     * {@link PayloadConverter}s, in this order. The map of payload converters
     * should contain the payload converter name as the key and the payload converter
     * instance as the value. Using the converter's name, the associated payload
     * converter in the map should be assigned to the converter that the publisher requires.
     * Payload converters required by the custom publisher should be correctly configured
     * within the custom publisher's configuration. The framework extracts the configured
     * payload converter information for the custom publisher from the service configuration,
     * constructs the map of payload converters, and supplies it to the constructor.
     * Hence, the supplied converters should be validated in the constructor to ensure
     * they are configured as required.
     * Below is an example of a constructor for a custom publisher that requires
     * payload converters:
     *
     * <pre>
     * {@code
     *     public KafkaCustomPublisher(
     *         Publisher publisher,
     *         Map<String, Object> configContext,
     *         Map<String, ? extends PayloadConverter<?, ?>> payloadConverterMap) {
     *     }
     * }
     * </pre>
     *
     * @param publisher the publisher instance that is used to delegate all
     *                  operations supported by the {@link Publisher}.
     */
    public CustomPublisher(Publisher publisher) {
        requireNonNull(publisher, "publisher must not be null");
        this.publisher = publisher;
    }

    /**
     * Processes the supplied update to publish to Diffusion topics.
     * <p>
     * The implementation of this method can call all the {@link Publisher}
     * methods via the abstract superclass.
     * <p>
     * The payload converter specified for the service will be applied to the
     * final value that is supplied to the
     * {@link Publisher#publish(String, Object, TopicProperties)} or
     * {@link Publisher#publish(String, Object)} method to convert it to a
     * Diffusion topic value. Hence, the implementation of this interface should
     * clarify what happens in this method and what type of updates will be
     * supplied to the
     * {@link Publisher#publish(String, Object, TopicProperties)} or
     * {@link Publisher#publish(String, Object)} methods.
     *
     * @param updateContext the Gateway application-specific and update-specific
     *                context that can be used to process the update. The
     *                context provided by different applications can be
     *                different. Hence, the custom publisher should handle them
     *                accordingly, and if any required context is absent, an
     *                {@link IllegalArgumentException} should be thrown. The
     *                application manual can be consulted to understand if it
     *                supports using a custom publisher and context it
     *                supplies.
     * @param update  the update to process.
     * @return a CompletableFuture that completes when processing the update
     *     completes.
     *     <p>
     *     To indicate success, the completable future should complete
     *     successfully with a null value. If a call to a {@link Publisher}
     *     method completes exceptionally then that future can be returned. If
     *     any other processing error occurs, the method should return an
     *     exceptionally completed future with a suitable exception.
     * @throws PayloadConversionException if the supplied value could not be
     *                                    converted by the payload converter
     *                                    configured for the service, or the
     *                                    value type is incompatible with the
     *                                    payload converter.
     */
    public abstract CompletableFuture<?> process(
        U update,
        C updateContext) throws PayloadConversionException;

    @Override
    public final CompletableFuture<?> publish(String path, Object value) throws PayloadConversionException {
        return publisher.publish(path, value);
    }

    @Override
    public final CompletableFuture<?> publish(String path, Object value,
        TopicProperties topicProperties) throws PayloadConversionException {
        return publisher.publish(path, value, topicProperties);
    }

    @Override
    public final CompletableFuture<?> applyJSONPatch(String path,
        String patch) throws IncompatibleConfigurationException {
        return publisher.applyJSONPatch(path, patch);
    }

    @Override
    public final CompletableFuture<?> remove(String topics) {
        return publisher.remove(topics);
    }

    @Override
    public final void setInitialJSONValueForPatchUpdates(
        String path,
        String jsonValue) {

        publisher.setInitialJSONValueForPatchUpdates(path, jsonValue);
    }

    @Override
    public final CompletableFuture<?> addMissingTopicHandler(
        String topicPath,
        MissingTopicNotificationHandler missingTopicNotificationHandler) {

        return
            publisher
                .addMissingTopicHandler(
                    topicPath,
                    missingTopicNotificationHandler);
    }

    @Override
    public final CompletableFuture<?> removeMissingTopicHandler(String topicPath) {
        return publisher.removeMissingTopicHandler(topicPath);
    }

    @Override
    public final TopicProperties getConfiguredTopicProperties() {
        return publisher.getConfiguredTopicProperties();
    }
}
