/*******************************************************************************
 * Copyright (c) 2023, 2024 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.diffusiondata.gateway.framework;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import io.micrometer.core.instrument.Meter;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Tag;

/**
 * Interface to supply {@link MeterRegistry} instance and to provide useful
 * utility methods to create tags which can be used by application and
 * framework.
 *
 * @author DiffusionData Limited
 */
public interface GatewayMeterRegistry {
    /**
     * Key for 'service' tag.
     * <p>
     * Tag with this name should be used for all service specific {@link
     * Meter}s. This is used to mark metrics for specific service.
     *
     * @see #getServiceNameTag(String)
     */
    String TAG_KEY_SERVICE = "service";

    /**
     * Key for 'reason' tag.
     * <p>
     * This can be used for creating tag for any meters where having a reason
     * makes sense.
     */
    String TAG_KEY_REASON = "reason";

    /**
     * Key for 'exception' tag.
     * <p>
     * This can be used for creating tag for any meters related to expose
     * exception related metrics.
     */
    String TAG_KEY_EXCEPTION = "exception";

    /**
     * Placeholder value for any tag.
     * <p>
     * This can be used as placeholder value for any tag, for which value is not
     * known beforehand and is updated when an event occurs.
     */
    String TAG_VALUE_PLACEHOLDER = "None";

    /**
     * Should return {@link MeterRegistry} instance for the application. This
     * can be any implementation of {@link MeterRegistry} for specific backend
     * monitoring tool or a composite registry for multiple monitoring tools.
     * More details about {@link MeterRegistry} can be found in official
     * documentation of
     * <a href="https://micrometer.io/docs/concepts">Micrometer</a>
     * <p>
     * Following the documentation of micrometer, to use {@link MeterRegistry}
     * for specific monitoring tool, corresponding dependency should be added
     * for application. For example: To expose application metrics in prometheus
     * following dependency should be added to maven's POM file of application.
     * <pre>
     * &lt;dependency&gt;
     *     &lt;groupId&gt;io.micrometer&lt;/groupId&gt;
     *     &lt;artifactId&gt;micrometer-registry-prometheus&lt;/artifactId&gt;
     *     &lt;version&gt;1.9.2&lt;/version&gt;
     * &lt;/dependency&gt;
     * </pre>
     * The returned {@link MeterRegistry} instance will be further configured by
     * the framework. If metrics is disabled in configuration, all registered
     * {@link Meter}s will be denied. Any meter names configured to be filtered
     * in configuration will be filtered out in MeterRegistry configuration.
     *
     * @return an instance of {@link MeterRegistry} for application specific
     *     monitoring tool.
     */
    MeterRegistry getMeterRegistry();

    /**
     * Creates aggregated list of tags with passed list and tag with exception
     * detail.
     *
     * @param tags      List of prebuilt tags to include.
     * @param exception Exception to fetch value for 'exception' tag.
     * @return List of tags with exception tag.
     */
    static List<Tag> createTagsWithException(
        List<Tag> tags,
        Throwable exception) {

        final String message = exception.getMessage();
        final String tagValue = message == null ?
            exception.getClass().getName() : message;

        return appendTags(tags, TAG_KEY_EXCEPTION, tagValue);
    }

    /**
     * Creates aggregated list of tags with passed key and value for a tag.
     *
     * @param tags  List of prebuilt tags to include.
     * @param key   Key name for tag.
     * @param value Value for tag.
     * @return List of tags with exception tag.
     */
    static List<Tag> appendTags(
        List<Tag> tags,
        String key,
        String value) {

        final List<Tag> finalTags = new ArrayList<>(tags);
        finalTags.add(Tag.of(key, value));

        return finalTags;
    }

    /**
     * Creates a singleton list with {@link Tag} containing name of the service
     * for which the metrics is being collected.
     * <p>
     * This method can be used to create a {@link Tag} with key 'service' and
     * passed 'serviceName' as value, which can be used with {@link Meter} for
     * services.
     *
     * @param serviceName to be used as value in tag.
     * @return a singleton list with {@link Tag} for specifying name of the
     *     service for which the metrics is being collected.
     */
    static List<Tag> getServiceNameTag(String serviceName) {
        return Collections.singletonList(Tag.of(TAG_KEY_SERVICE, serviceName));
    }
}
