/*******************************************************************************
 * Copyright (c) 2024 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.diffusiondata.gateway.framework;

/**
 * Diffusion topic properties.
 *
 * @author DiffusionData Limited
 * @since 2.0
 */
public interface TopicProperties {

    /**
     * Returns the topic type.
     * <p>
     *
     * @return the topic type
     * @see TopicProperties#withTopicType(TopicType)
     */
    TopicType getTopicType();

    /**
     * Returns the boolean value to specify if the topic is a time series
     * topic.
     * <p>
     *
     * @return the boolean value to specify if the topic is a time series topic
     * @see TopicProperties#withTimeSeries(boolean)
     */
    boolean isTimeSeries();

    /**
     * Returns the persistence policy for the topic.
     * <p>
     *
     * @return the persistence policy for the topic
     * @see TopicProperties#withPersistencePolicy(PersistencePolicy)
     */
    PersistencePolicy getPersistencePolicy();

    /**
     * Returns a retained range for a time series topic.
     * <p>
     *
     * @return a retained range for a time series topic
     * @see TopicProperties#withTimeSeriesRetainedRange(String)
     */
    String getTimeSeriesRetainedRange();

    /**
     * Returns a subscription range for a time series topic.
     * <p>
     *
     * @return a subscription range for a time series topic
     * @see TopicProperties#withTimeSeriesSubscriptionRange(String)
     */
    String getTimeSeriesSubscriptionRange();

    /**
     * Returns a boolean value to specify whether a topic should publish only
     * values.
     * <p>
     *
     * @return a boolean value to specify whether a topic should publish only
     *     values
     * @see TopicProperties#withPublishValuesOnly(boolean)
     */
    boolean publishValuesOnly();

    /**
     * Returns a boolean value to specify whether a topic should not retain its
     * last value.
     * <p>
     *
     * @return a boolean value to specify whether a topic should not retain its
     *     last value
     * @see TopicProperties#withDontRetainValue(boolean)
     */
    boolean dontRetainValue();

    /**
     * Returns a new instance of TopicProperties with the supplied topic type and
     * other existing properties.
     * <p>
     * The supplied value will replace the configured or default
     * topicType, which is {@link TopicType#JSON}.
     * <p>
     *
     * @param topicType the topic type
     *
     * @return a new instance of TopicProperties with the supplied topic type
     */
    TopicProperties withTopicType(TopicType topicType);

    /**
     * Returns a new instance of TopicProperties with the supplied boolean value to
     * specify if the topic is time series and other existing properties.
     * <p>
     * The supplied value will replace the configured or default
     * timeSeriesRetainedRange, which is null.
     * <p>
     * If the topic is not a time series topic, then the
     * `timeSeriesRetainedRange` and `timeSeriesSubscriptionRange` specified
     * for the TopicProperties will be ignored when creating the topic.
     * <p>
     * @param isTimeSeries a boolean value to specify if the topic is a time
     *                     series topic
     * @return a new instance of TopicProperties with the supplied boolean
     * value to specify if the topic is time series
     */
    TopicProperties withTimeSeries(boolean isTimeSeries);

    /**
     * Returns a new instance of TopicProperties with the supplied PersistencePolicy
     * and other existing properties.
     * <p>
     * The supplied value will replace the configured or default
     * persistencePolicy, which is {@link PersistencePolicy#SESSION}.
     * <p>
     *
     * @param persistencePolicy the persistence policy for topic
     *
     * @return a new instance of TopicProperties with the supplied PersistencePolicy
     */
    TopicProperties withPersistencePolicy(PersistencePolicy persistencePolicy);

    /**
     * Returns a new instance of TopicProperties with the supplied
     * timeSeriesRetainedRange and other existing properties.
     * <p>
     * The `timeSeriesRetainedRange` parameter configures the range of historic
     * events retained by a time series topic.
     * <p>
     * When a new event is added to the time series, older events that fall
     * outside the range are discarded.
     * <p>
     * The parameter value is a time series range expression string composed of
     * one or more constraint clauses. Constraints are combined to provide a
     * range of events from the end of the time series.
     * <p>
     * <b>limit constraint:</b> A limit constraint specifies the maximum
     * number of events from the end of the time series.
     * <p>
     * <b>last clause</b>: A last constraint specifies the maximum duration
     * of events from the end of the time series. The duration is expressed as
     * an integer followed by one of the following time units. MS -
     * milliseconds; S - seconds; H - hours.
     * <p>
     * If a range expression contains multiple constraints, the constraint that
     * selects the smallest range is used.
     * <br>
     * <table border="1">
     * <caption><b>Retained Range expressions</b></caption>
     * <tr>
     * <th>Parameter value</th>
     * <th>Meaning</th>
     * </tr>
     * <tr>
     * <td>limit 5</td>
     * <td>The five most recent events</td>
     * </tr>
     * <tr>
     * <td>last 10s</td>
     * <td>All events that are no more than ten seconds older than the
     * latest event</td>
     * </tr>
     * <tr>
     * <td>last 10s limit 5</td>
     * <td>The five most recent events that are no more than ten seconds
     * older than the latest event</td>
     * </tr>
     * </table>
     * <p>
     * Range expressions are not case sensitive: `limit 5 last 10s` is
     * equivalent to `LIMIT 5 LAST 10S`.
     * <p>
     *
     * The supplied value will replace the configured or default
     * timeSeriesRetainedRange, which is null. If a null value is used, a
     * time series topic will retain the ten most recent events.
     *
     * @param timeSeriesRetainedRange the timeSeriesRetainedRange for topic
     * @return a new instance of TopicProperties with the supplied
     *     timeSeriesRetainedRange
     */
    TopicProperties withTimeSeriesRetainedRange(String timeSeriesRetainedRange);

    /**
     * Returns a new instance of TopicProperties with the supplied
     * timeSeriesSubscriptionRange and other existing properties.
     * <p>
     * The `timeSeriesSubscriptionRange` parameter configures a time series
     * topic to send a range of historic events from the end of the time series
     * to new subscribers.
     * <p>
     * Configuring a subscription range is a convenient way to provide new
     * subscribers with an appropriate subset of the latest events.
     * <p>
     * A larger subscription range can be configured using the
     * `subscriptionRange` parameter. Regardless of the parameter setting, if
     * delta streams are enabled, new subscribers will be sent at least the
     * latest event if one exists.
     * <p>
     * If the range of events is insufficient, the subscribing session can use a
     * range query to retrieve older events.
     * <p>
     * The value for this property is a time series range expression, following
     * the format used for `timeSeriesRetainedRange`.
     * <p>
     * The supplied value will replace the configured or default
     * timeSeriesSubscriptionRange, which is null. If a null value is used,
     * the default subscription range is used. The default subscription range
     * depends on whether the topic is configured to publish delta streams.
     * If delta streams are enabled, new subscribers are sent the latest
     * event if one exists. If delta streams are disabled, new subscribers
     * are sent no events. Delta streams are enabled by default and can be
     * disabled by setting `publishValuesOnly` to `true`.
     * <p>
     *
     * @param timeSeriesSubscriptionRange the timeSeriesSubscriptionRange for
     *                                    topic
     * @return a new instance of TopicProperties with the supplied
     *     timeSeriesSubscriptionRange
     */
    TopicProperties withTimeSeriesSubscriptionRange(String timeSeriesSubscriptionRange);

    /**
     * Returns a new instance of TopicProperties with the supplied boolean value to
     * specify whether a topic should publish only values.
     * <p>
     * By default, a topic that supports delta streams will publish the
     * difference between two values (a delta) when doing so is more efficient
     * than publishing the complete new value.
     * <p>
     * Setting `publishValuesOnly` to `true` disables this behaviour so that
     * deltas are never published. Doing so is usually not recommended because
     * it will result in more data being transmitted, less efficient use of
     * network resources, and increased transmission latency. On the other hand,
     * calculating deltas can require significant CPU from the server or, if
     * update streams are used, from the updating client. The CPU cost will be
     * higher if there are many differences between successive values, in which
     * case delta streams confer fewer benefits. If successive values are
     * unrelated to each other, consider setting to `true`. Also consider
     * setting to `true` if the network capacity is high and the bandwidth
     * savings of deltas are not required.
     * <p>
     * The supplied boolean value will replace the configured or default
     * value for `publishValuesOnly`, which is `false`.
     * <p>
     *
     * @param publishValuesOnly the boolean value to specify whether to publish
     *                          only values
     * @return a new instance of TopicProperties with the supplied boolean value to
     *     specify whether a topic should publish only values
     */
    TopicProperties withPublishValuesOnly(boolean publishValuesOnly);

    /**
     * Returns a new instance of TopicProperties with the supplied boolean value to
     * specify whether a topic should not retain its last value.
     * <p>
     * By default, a topic will retain its latest value. The latest value will
     * be sent to new subscribers. Setting this property to `true` disables this
     * behaviour. New subscribers will not be sent an initial value. No value
     * will be returned for fetch operations that select the topic. This is
     * useful for data streams where the values are only transiently valid.
     * <p>
     * Setting to `true` also disables delta streams, regardless of the
     * `publishValuesOnly` value. If subsequent values are likely to be related,
     * delta streams usually provide performance benefits (see
     * `publishValuesOnly` above). Consider leaving as `false` to benefit from
     * delta streams, even if there is no other requirement to retain the last
     * value.
     * <p>
     * Bearing in mind the performance trade-offs of disabling delta streams,
     * there are two reasons to consider setting to `true`. First, it stops the
     * server and each subscribed client from keeping a copy of the value,
     * reducing their memory requirements. Second, when a topic has a high
     * update rate and is replicated across a cluster, it can significantly
     * improve throughput because the values need not be persisted to the
     * cluster.
     * <p>
     * The supplied boolean value will replace the configured or default
     * value for `dontRetainValue`, which is `false`.
     * <p>
     *
     * @param dontRetainValue the boolean value to specify whether a topic
     *                        should not retain its last value
     * @return a new instance of TopicProperties with the supplied boolean value to
     *     specify whether a topic should not retain its last value
     */
    TopicProperties withDontRetainValue(boolean dontRetainValue);

    /**
     * The persistence policy to be applied to topics created by the service.
     */
    enum PersistencePolicy {

        /**
         * Service level persistence.
         * <p>
         * Topics created and published by the service will be removed from the
         * Diffusion server whenever the service is paused and will be recreated
         * when published to again after resuming.
         * <p>
         * Such topics will also be removed when the framework connection with
         * Diffusion is lost (causing a pause) or when the application closes.
         * <p>
         * If the service is paused while topics are being published, such
         * topics will not be removed, as pause attempts to only remove topics
         * which have already been published by the service.
         * <p>
         * If the total number of created topics exceeds the maximum cache size,
         * configured with the 'gateway.cache.max.size' system property (default
         * value: 1000), the service will only remove the topics stored in the
         * local cache.
         */
        SERVICE,

        /**
         * Session level persistence.
         * <p>
         * Topics created by the service will be removed from the Diffusion
         * server when the framework connection to Diffusion is lost or when the
         * application closes.
         */
        SESSION,

        /**
         * Server level persistence.
         * <p>
         * Topics created by the service persist regardless of disconnections
         * and after the application closes, as long as the Diffusion server is
         * running.
         */
        SERVER,

        /**
         * Full persistence.
         * <p>
         * Topics created by the service will persist indefinitely even beyond
         * the life of the Diffusion server instance.
         * <p>
         * Such topics would be restored when the Diffusion server restarts and
         * can only be removed by some other mechanism (such as using the {@link
         * Publisher#remove Publisher.remove} method or using the Diffusion
         * Console or client APIs.
         * <p>
         * This policy will only work if the Diffusion server is configured to
         * support persistence, otherwise it is the same as {@link #SERVER}.
         */
        FULL
    }
}
