/*******************************************************************************
 * Copyright (c) 2022 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.diffusiondata.gateway.framework.exceptions;

import java.util.Collections;
import java.util.Map;

import com.diffusiondata.gateway.framework.GatewayApplication;

/**
 * Exception thrown by a {@link GatewayApplication} to indicate that the
 * configuration parameters that have been passed to it are invalid.
 *
 * @author DiffusionData Limited
 */
public final class InvalidConfigurationException
    extends GatewayApplicationException {

    private static final long serialVersionUID = -2655519979304900318L;

    private final Map<String, String> errors;

    /**
     * Constructor for a general purpose message.
     *
     * @param message the message
     */
    public InvalidConfigurationException(String message) {
        super(message);
        errors = null;
    }

    /**
     * Constructor for a general purpose message and specific throwable.
     *
     * @param message the message
     * @param cause cause for the exception
     */
    public InvalidConfigurationException(String message, Throwable cause) {
        super(message, cause);
        errors = null;
    }

    /**
     * Constructor for multiple errors.
     *
     * @param serviceName the service name to use to append in the message
     * @param errors      a map of key/message pairs, where the key identifies
     *                    the configuration item in error and the message
     *                    describes the error
     */
    public InvalidConfigurationException(
        String serviceName, Map<String, String> errors) {

        super(
            "Invalid configuration found for service '" + serviceName + "'");
        this.errors = Collections.unmodifiableMap(errors);
    }

    /**
     * Returns the map of errors.
     *
     * @return map of errors
     */
    public Map<String, String> getErrors() {
        return errors == null ? Collections.emptyMap() : errors;
    }

    @Override
    public String getMessage() {
        if (errors == null) {
            return super.getMessage();
        }
        else {
            return super.getMessage() + " : " + errors;
        }
    }

}
